C> \ingroup wfn1
C> @{
C>
C> \brief Calculate the energy and the analytical gradient in WFN1
C> formalism
C>
C> To date the method has been implemented using numerical derivatives.
C> This was convenient for testing the basics and easy to program.
C> For serious computation numerically differentiating the energy
C> expression is far too slow. Hence this routine implements the 
C> analytical derivatives of the energy expression.
C>
C> Summarizing the terminology, we have natural orbitals expressed in
C> terms of the atomic orbitals (AO)
C> \f{eqnarray*}{
C>   \phi^{N\sigma}_i &=& \sum_a \chi_a(r) C^{N\sigma}_{ai} \\\\
C>   (\phi^{N\sigma}_i|\phi^{N\sigma}_j) &=& \delta_{ij}
C> \f}
C> as usual the natural orbitals are eigenfunctions of the 1-electron
C> density matrix and hence form an orthonormal set (for each 
C> N-electron state).
C>
C> In addition we now also have occupation functions expressed in terms
C> of natural orbitals (NO)
C> \f{eqnarray*}{
C>   \phi^{O\sigma}_p &=& \sum_i \phi^{N\sigma}_i C^{O\sigma}_{ip} \\\\
C>   (\phi^{O\sigma}_p|\phi^{O\sigma}_q) &=& \delta_{pq}
C> \f}
C> Just like the natural orbitals the occupation functions form an
C> orthonormal set for each N-electron state. In addition the square of
C> the occupation function gives the fraction that a given electron
C> contributes to the total occupation number of a natural orbital.
C> Therefore the occupation numbers of the natural orbitals can be
C> written as
C> \f{eqnarray*}{
C>    d^\sigma_{i} 
C>    &=& \sum_{p=1}^{N^\sigma_{el}}
C>    \left[\left(\phi^{O\sigma}_p\right)^*\phi^{O\sigma}_p\right]_i \\\\
C>    &=& \sum_{p=1}^{N^\sigma_{el}}
C>    \left(C^{O\sigma}_{ip}\right)^* C^{O\sigma}_{ip}
C> \f}
C> Given the natural orbitals and the occupation functions an extended
C> orbital can be written as
C> \f{eqnarray*}{
C>   \phi^{X\sigma}_p &=& \{\phi^{N\sigma}\}\times\phi^{O\sigma}_p
C> \f}
C> Written out in full the extended orbital has \f$O(N^2)\f$ 
C> coefficients (even though the space is only \f$N\f$-dimensional).
C>
C> The norm in this space can be defined in terms of the usual overlap
C> matrix by replicating it \f$N\f$ times on each row and \f$N\f$ times
C> down each column
C> \f{eqnarray*}{
C>   \delta_{pq} &=& \left(\phi^{X\sigma}_p\right)^T\left(
C>   \begin{array}{ccc}
C>      \mathbf{S} & \cdots & \mathbf{S} \\\\
C>      \vdots     & \ddots & \vdots     \\\\
C>      \mathbf{S} & \cdots & \mathbf{S} \\\\
C>   \end{array}\right)
C>   \phi^{X\sigma}_q
C> \f}
C> In practice it is usually smarter to work with natural orbitals 
C> and occupation functions rather than writing extended orbitals out
C> in full.
C>
C> Having defined these extended orbitals it is possible to define
C> a wavefunction of \f$N^\alpha_{el}\f$ and \f$N^\beta_{el}\f$ 
C> \f$\alpha\f$- and \f$\beta\f$-electrons respectively, as
C> \f{eqnarray*}{
C>   \Psi_I &=& 
C>   \left|\phi^{X\alpha}_1\cdots\phi^{X\alpha}_{N^\alpha_{el}}\right|
C>   \left|\phi^{X\beta}_1\cdots\phi^{X\beta}_{N^\beta_{el}}\right|
C> \f}
C> I.e. this wavefunction is simply a single Slater determinant 
C> wavefunction. It would be identical to the Hartree-Fock wavefunction
C> if conventional orbitals (or equivalently natural orbitals) were
C> used instead of extended orbitals.
C>
C> The 1-electron density matrix generated by this wavefunction is
C> given by
C> \f{eqnarray*}{
C>   D^\sigma_{ab}
C>   &=& \sum_{p=1}^{N^\sigma_{el}}
C>              \left|\phi^{X\sigma}_p\right\rangle
C>              \left\langle\phi^{X\sigma}_p\right| \\\\
C>   &=& \sum_i\sum_{p=1}^{N^\sigma_{el}}
C>       C^{N\sigma}_{ai}C^{O\sigma}_{ip}
C>       C^{O\sigma}_{ip}C^{N\sigma}_{bi}
C> \f}
C> It is easy to verify that this density matrix is identical to the
C> FullCI 1-electron density matrix if suitable coefficients are
C> chosen for the natural orbitals and the occupation functions.
C>
C> It is possible to use a wavefunction of this form to calculate
C> excited states by insisting that it be orthogonal to all lower
C> wavefunctions
C> \f{eqnarray*}{
C>   \forall_{J<I},\; \left\langle\Psi_J\right.\left|\Psi_I\right\rangle
C>   = 0
C> \f}
C> In regular effective 1-electron models the energy of a given
C> state can be expressed in terms of the 1-electron density matrices.
C> Adding the appropriate Langrangian constraints we have for the
C> energy of an arbitrary state 
C> \f{eqnarray*}{
C>    E_I &=& E(D^\alpha,D^\beta)
C>        + \sum_\sigma\lambda^{O\sigma}\left(\sum_iC^{O\sigma}_{ip}
C>          C^{O\sigma}_{iq}-\delta_{pq}\right) \\\\
C>       &+&\sum_\sigma\lambda^{N\sigma}\left(\sum_{ab}C^{N\sigma}_{ai}
C>          S_{ab}C^{N\sigma}_{bj}-\delta_{ij}\right)
C>        + \sum_{J<I}\Lambda_J\left\langle\Psi_J\right.
C>          \left|\Psi_I\right\rangle
C> \f}
C> Here all density matrices and coefficients are assumed to correspond
C> to state \f$I\f$ unless indicated otherwise.
C>
C> In practice, however, there is a problem. The density matrix has only
C> \f$N^2\f$ variables, but the wavefunction has \f$2N^2\f$ variables.
C> I.e. there are a number of redundant variables. This is similar to
C> the redundant variables in CASSCF, for example. The problem with this
C> is that there are rotations among the redundant variables that 
C> ensure orthogonality between two states that represent nearly the
C> same density matrices. I.e. the redundant coordinates weaken the
C> orthonormality constraints to the point that unphysical results
C> are admitted. Thus looking for excited states by imposing 
C> orthonormality in this space is not a feasible approach. The
C> remaining alternative is to look for states that are stationary
C> points of the energy expression. That is, we consider an N-th fold
C> excited state to be an N-th order transition state of the energy
C> expression with respect to the density matrix parameters. To find 
C> these points we first guess the state by permuting occupied and
C> virtual orbitals (swapping two orbitals corresponds to a 180
C> degree rotation among them). Subsequently we can use a Newton-
C> Raphson algorithm to find the point.
C>
C> Differentiating the first term with respect to the occupation
C> function coefficients we obtain
C> \f{eqnarray*}{
C>   \frac{\partial E}{\partial C^{O\sigma}_{ip}} &=&
C>   \sum_{ab}
C>   \frac{\partial E}{\partial D^{\sigma}_{ab}}
C>   \frac{\partial D^{\sigma}_{ab}}{\partial C^{O\sigma}_{ip}} \\\\
C>   &=& 2\sum_{ab}\sum_i F^\sigma_{ab}C^{N\sigma}_{aj}C^{N\sigma}_{bi}
C>       C^{O\sigma}_{ip}
C> \f}
C> I.e. the Fock-matrix for the occupation functions is simply the
C> Fock-matrix in AO basis transformed with the natural orbitals.
C>
C> Differentiating the first term with respect to the natural orbital
C> coefficients we obtain
C> \f{eqnarray*}{
C>   \frac{\partial E}{\partial C^{N\sigma}_{cj}} &=&
C>   \sum_{ab}
C>   \frac{\partial E}{\partial D^{\sigma}_{ab}}
C>   \frac{\partial D^{\sigma}_{ab}}{\partial C^{N\sigma}_{cj}} \\\\
C>   &=& \sum_{ab} F^\sigma_{ab} \sum_{ip}\left(
C>       \delta_{ca}\delta_{ij}C^{N\sigma}_{bi}
C>       C^{O\sigma}_{ip}C^{O\sigma}_{ip}+
C>       \delta_{cb}\delta_{ij}C^{N\sigma}_{ai}
C>       C^{O\sigma}_{ip}C^{O\sigma}_{ip}\right) \\\\
C>   &=& \sum_{ab} F^\sigma_{ab} \sum_{ip}
C>       \delta_{ca}\delta_{ij}C^{N\sigma}_{bi}
C>       C^{O\sigma}_{ip}C^{O\sigma}_{ip}+
C>       \sum_{ab} F^\sigma_{ab} \sum_{ip}
C>       \delta_{cb}\delta_{ij}C^{N\sigma}_{ai}
C>       C^{O\sigma}_{ip}C^{O\sigma}_{ip} \\\\
C>   &=& 2\sum_b F^\sigma_{ab}C^{N\sigma}_{bj}
C>       \sum_p C^{O\sigma}_{jp}C^{O\sigma}_{jp}
C> \f}
C> I.e. we arrive at a Fock matrix that is weighted with the 
C> orbital occupation numbers. This matches the traditional
C> Hartree-Fock matrix in as far as the occupied orbitals are concerned.
C>
      subroutine wfn1_engrad_ana(rtdb,geom,nst,nbf,nea,neb,h1,eri,erix,
     &                           ov,nperma,npermb,perma,permb,
     &                           signa,signb,r_noa,r_nob,r_ofa,r_ofb,
     &                           En,enoa,enob,eofa,eofb,
     &                           d_noa,d_nob,d_ofa,d_ofb,
     &                           ovla,ovlb,focka,fockb,tfock)
      implicit none
c
#include "errquit.fh"
#include "geom.fh"
c
      integer rtdb !< [Input] The RTDB handle
      integer geom !< [Input] The geometry handle
      integer nbf  !< [Input] The number of basis functions
      integer nst  !< [Input] The number of states for which
                   !< energies and gradients should be evaluated
      integer nea  !< [Input] The number of \f$\alpha\f$-spin
                   !< electrons for each state
      integer neb  !< [Input] The number of \f$\beta\f$-spin
                   !< electrons for each state
      integer nperma !< [Input] The number alpha electron permutations
      integer npermb !< [Input] The number beta electron permutations
      integer perma(nea,nperma) !< [Input] The alpha electron
                                !< permutations
      integer permb(neb,npermb) !< [Input] The beta electron
                                !< permutations
c
      double precision h1(nbf,nbf) !< [Input] The 1-electron integrals
      double precision eri(nbf,nbf,nbf,nbf) !< [Input] The 2-electron
                                            !< integrals
      double precision erix(nbf,nbf,nbf,nbf) !< [Input] The 2-electron
                                             !< integrals for exchange
      double precision ov(nbf,nbf) !< [Input] The overlap integrals
c
      double precision r_noa(nbf,nbf,nst) !< [Input] The
      !< \f$\alpha\f$-spin natural orbitals. This is a 3D array with the
      !< dimensions:
      !< - 1. The basis functions
      !< - 2. The natural orbitals
      !< - 3. The states
      double precision r_nob(nbf,nbf,nst) !< [Input] The
      !< \f$\beta\f$-spin natural orbitals. This is a 3D array with the
      !< same dimensions as `r_noa`.
      double precision r_ofa(nbf,nbf,nst) !< [Input] The
      !< \f$\alpha\f$-spin occupation functions. This is a 3D array with
      !< the dimension:
      !< - 1. The natural orbitals
      !< - 2. The occupation functions
      !< - 3. The states
      double precision r_ofb(nbf,nbf,nst) !< [Input] The
      !< \f$\beta\f$-spin occupation functions. This is a 3D array with
      !< the same dimensions as `r_ofa`.
      double precision signa(nperma) !< [Input] The sign of the alpha
                                     !< permutations
      double precision signb(npermb) !< [Input] The sign of the beta
                                     !< permutations

c
      double precision En !< [Output] The total energy
      double precision enoa(nbf) !< [Output] The \f$\alpha\f$ natural
      !< orbital energies
      double precision enob(nbf) !< [Output] The \f$\beta\f$ natural
      !< orbital energies
      double precision eofa(nbf) !< [Output] The \f$\alpha\f$ occupation
      !< function energies
      double precision eofb(nbf) !< [Output] The \f$\beta\f$ occupation
      !< function energies
c
      double precision d_noa(nbf,nbf) !< [Output] The \f$\alpha\f$
      !< natural orbital gradient given in the natural orbital basis
      double precision d_nob(nbf,nbf) !< [Output] The \f$\beta\f$
      !< natural orbital gradient given in the natural orbital basis
      double precision d_ofa(nbf,nbf) !< [Output] The \f$\alpha\f$
      !< occupation function gradient given in the occupation function
      !< basis
      double precision d_ofb(nbf,nbf) !< [Output] The \f$\beta\f$
      !< occupation function gradient given in the occupation function
      !< basis
c
      double precision ovla(nea,nea) !< [Scratch] The alpha overlaps
      double precision ovlb(neb,neb) !< [Scratch] The beta overlaps
      double precision focka(nbf,nbf) !< [Scratch] The \f$\alpha\f$
                                      !< electron Fock matrix
      double precision fockb(nbf,nbf) !< [Scratch] The \f$\beta\f$
                                      !< electron Fock matrix
      double precision densa(nbf,nbf) !< [Scratch] The \f$\alpha\f$
                                      !< electron density matrix
      double precision densb(nbf,nbf) !< [Scratch] The \f$\beta\f$
                                      !< electron density matrix
      double precision docca(nbf)     !< [Scratch] The \f$\alpha\f$
                                      !< electron occupation numbers
      double precision doccb(nbf)     !< [Scratch] The \f$\beta\f$
                                      !< electron occupation numbers
      double precision tfock(nbf,nbf) !< [Scratch] Fock matrix
      double precision aa             !< [Scratch]
c
c     Local
c
      integer i, j, p, q, a, b
      double precision Enuc  !< The nuclear repulsion energy
      double precision E1el  !< The 1-electron energy
      double precision E2elc !< The 2-electron Coulomb energy
      double precision E2elx !< The 2-electron exchange energy
      double precision Exc   !< The exchange-correlation energy
      double precision hfwght!< The weight of the Hartree-Fock exchange
c
      double precision ddot, xc_hfexch
      external         ddot, xc_hfexch
c
c     The nuclear repulsion term
c
      if (.not.geom_nuc_rep_energy(geom,Enuc))
     &  call errquit("wfn1_energy: Enuc failed",0,GEOM_ERR)
c
c     Construct the 1-electron density matrices
c
      do i = 1, nbf
        docca(i) = 0.0d0
        doccb(i) = 0.0d0
        do p = 1, nea
          docca(i) = docca(i) + r_ofa(i,p,nst)**2
          doccb(i) = doccb(i) + r_ofb(i,p,nst)**2
        enddo
      enddo
      call dfill(nbf*nbf,0.0d0,densa,1)
      call dfill(nbf*nbf,0.0d0,densb,1)
      do a = 1, nbf
        do b = 1, nbf
          do i = 1, nbf
            densa(b,a) = densa(b,a)
     &                 + r_noa(a,i,nst)*r_noa(b,i,nst)*docca(i)
            densb(b,a) = densb(b,a)
     &                 + r_nob(a,i,nst)*r_nob(b,i,nst)*doccb(i)
          enddo
        enddo
      enddo
c
c     The 1-electron term
c
      call dcopy(nbf*nbf,h1,1,focka,1)
      call dcopy(nbf*nbf,h1,1,fockb,1)
      E1el = ddot(nbf*nbf,focka,1,densa,1)
     &     + ddot(nbf*nbf,fockb,1,densb,1)
c
c     The 2-electron Coulomb term
c
      E2elc = 0.0d0
      call dgemv('N',nbf*nbf,nbf*nbf,1.0d0,eri,nbf*nbf,densa,1,
     +           0.0d0,tfock,1)
      E2elc = E2elc + ddot(nbf*nbf,tfock,1,densa,1)/2.0d0
      E2elc = E2elc + ddot(nbf*nbf,tfock,1,densb,1)/2.0d0
      call daxpy(nbf*nbf,1.0d0,tfock,1,focka,1)
      call daxpy(nbf*nbf,1.0d0,tfock,1,fockb,1)
      call dgemv('N',nbf*nbf,nbf*nbf,1.0d0,eri,nbf*nbf,densb,1,
     +           0.0d0,tfock,1)
      E2elc = E2elc + ddot(nbf*nbf,tfock,1,densa,1)/2.0d0
      E2elc = E2elc + ddot(nbf*nbf,tfock,1,densb,1)/2.0d0
      call daxpy(nbf*nbf,1.0d0,tfock,1,fockb,1)
      call daxpy(nbf*nbf,1.0d0,tfock,1,focka,1)
c
c     The 2-electron exchange term
c
      hfwght = xc_hfexch()
      E2elx  = 0.0d0
      call dgemv('N',nbf*nbf,nbf*nbf,hfwght,0.0d0,erix,nbf*nbf,densa,1,
     +           0.0d0,tfock,1)
      E2elx = E2elx - ddot(nbf*nbf,densa,1,tfock,1)/2.0d0
      call daxpy(nbf*nbf,-1.0d0,tfock,1,focka,1)
      call dgemv('N',nbf*nbf,nbf*nbf,hfwght,0.0d0,erix,nbf*nbf,densb,1,
     +           0.0d0,tfock,1)
      E2elx = E2elx - ddot(nbf*nbf,densb,1,tfock,1)/2.0d0
      call daxpy(nbf*nbf,-1.0d0,tfock,1,fockb,1)
c
c     The DFT exchange-correlation term
c
      call wfn1_xc_f(rtdb,nbf,nea,neb,densa,densb,Exc,focka,fockb,tfock)
c
c     Done constructing Fock matrices.
c     Now construct the appropriate gradients
c
c     1. Multiply alpha N^T*F where N are the natural orbitals and F
c        is the Fock-matrix, we can use the result twice
c
      call dgemm('T','N',nbf,nbf,nbf,1.0d0,r_noa(1,1,nst),nbf,
     +           focka,nbf,0.0d0,tfock,nbf)
c
c     use the space for the occupation function gradients as scratch
c
      do i = 1, nbf
        do a = 1, nbf
          d_ofa(a,i) = r_noa(a,i,nst)*docca(i)
        enddo
      enddo
      call dgemm('N','N',nbf,nbf,nbf,1.0d0,tfock,nbf,d_ofa,nbf,0.0d0,
     +           d_noa,nbf)
      do j = 1, nbf
        do i = j+1, nbf
          aa = 0.5d0*(d_noa(i,j)+d_noa(j,i))
          d_noa(i,j) =  aa
          d_noa(j,i) = -aa
        enddo
        enoa(j) = d_noa(j,j)
        d_noa(j,j) = 0.0d0
      enddo
c
c     now do the other transformation of tfock using once again
c     the occupation function gradients as scratch
c
      call dgemm('N','N',nbf,nbf,nbf,1.0d0,tfock,nbf,r_noa(1,1,nst),nbf,
     +           0.0d0,d_ofa,nbf)
      call dgemm('T','N',nbf,nbf,nbf,1.0d0,r_ofa(1,1,nst),nbf,d_ofa,nbf,
     +           0.0d0,tfock,nbf)
      call dgemm('N','N',nbf,nbf,nbf,1.0d0,tfock,nbf,r_ofa(1,1,nst),nbf,
     +           0.0d0,d_ofa,nbf)
      do q = 1, nbf
        do p = q+1, nbf
          d_ofa(q,p) = -d_ofa(p,q)
        enddo
        eofa(q) = d_ofa(q,q)
        d_ofa(q,q) = 0.0d0
      enddo
c
c     2. Multiply beta N^T*F where N are the natural orbitals and F
c        is the Fock-matrix, we can use the result twice
c
      call dgemm('T','N',nbf,nbf,nbf,1.0d0,r_nob(1,1,nst),nbf,
     +           fockb,nbf,0.0d0,tfock,nbf)
c
c     use the space for the occupation function gradients as scratch
c
      do i = 1, nbf
        do a = 1, nbf
          d_ofb(a,i) = r_nob(a,i,nst)*doccb(i)
        enddo
      enddo
      call dgemm('N','N',nbf,nbf,nbf,1.0d0,tfock,nbf,d_ofb,nbf,0.0d0,
     +           d_nob,nbf)
      do j = 1, nbf
        do i = j+1, nbf
          aa = 0.5d0*(d_nob(i,j)+d_nob(j,i))
          d_nob(i,j) =  aa
          d_nob(j,i) = -aa
        enddo
        enob(j) = d_nob(j,j)
        d_nob(j,j) = 0.0d0
      enddo
c
c     now do the other transformation of tfock using once again
c     the occupation function gradients as scratch
c
      call dgemm('N','N',nbf,nbf,nbf,1.0d0,tfock,nbf,r_nob(1,1,nst),nbf,
     +           0.0d0,d_ofb,nbf)
      call dgemm('T','N',nbf,nbf,nbf,1.0d0,r_ofb(1,1,nst),nbf,d_ofb,nbf,
     +           0.0d0,tfock,nbf)
      call dgemm('N','N',nbf,nbf,nbf,1.0d0,tfock,nbf,r_ofb(1,1,nst),nbf,
     +           0.0d0,d_ofb,nbf)
      do q = 1, nbf
        do p = q+1, nbf
          d_ofb(q,p) = -d_ofb(p,q)
        enddo
        eofb(q) = d_ofb(q,q)
        d_ofb(q,q) = 0.0d0
      enddo
c
      end
C> @}
