/*
** (c) 1996-2000 The Regents of the University of California (through
** E.O. Lawrence Berkeley National Laboratory), subject to approval by
** the U.S. Department of Energy.  Your use of this software is under
** license -- the license agreement is attached and included in the
** directory as license.txt or you may contact Berkeley Lab's Technology
** Transfer Department at TTD@lbl.gov.  NOTICE OF U.S. GOVERNMENT RIGHTS.
** The Software was developed under funding from the U.S. Government
** which consequently retains certain rights as follows: the
** U.S. Government has been granted for itself and others acting on its
** behalf a paid-up, nonexclusive, irrevocable, worldwide license in the
** Software to reproduce, prepare derivative works, and perform publicly
** and display publicly.  Beginning five (5) years after the date
** permission to assert copyright is obtained from the U.S. Department of
** Energy, and subject to any subsequent five (5) year renewals, the
** U.S. Government is granted for itself and others acting on its behalf
** a paid-up, nonexclusive, irrevocable, worldwide license in the
** Software to reproduce, prepare derivative works, distribute copies to
** the public, perform publicly and display publicly, and to permit
** others to do so.
*/


//
// $Id: AmrDerive_Vfluct.cpp,v 1.2 2003/01/16 21:21:28 marc Exp $
//

#include <winstd.H>
#include <new>
#include <iostream>
#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <fstream>

#include <unistd.h>

#include "REAL.H"
#include "Box.H"
#include "FArrayBox.H"
#include "ParmParse.H"
#include "ParallelDescriptor.H"
#include "DataServices.H"
#include "Utility.H"
#include "VisMF.H"
#include "Derived.H"

#include "ArrayLim.H"

//
// This MUST be defined if don't have pubsetbuf() in I/O Streams Library.
//
#ifdef BL_USE_SETBUF
#define pubsetbuf setbuf
#endif

#if defined(BL_FORT_USE_UPPERCASE)
#  define TRANSV  TRANSV
#  define NORMV   NORMV
#elif defined(BL_FORT_USE_LOWERCASE)
#  define TRANSV  transv
#  define NORMV   normv
#elif defined(BL_FORT_USE_UNDERSCORE)
#  define TRANSV  transv_
#  define NORMV   normv_
#endif

extern "C" {
    void TRANSV(Real* v, ARLIM_P(vlo), ARLIM_P(vhi),
                const Real* dx, const Real* domnlo,
                const Real* avVR, const Real* avVT, const Real* avVZ, Real* avC,
                int* Nbin,
                ARLIM_P(alo), ARLIM_P(ahi), const int* maxbin, const int* bsize,
                Real* rf,  Real* tf, Real* zf, Real* cf);

    void NORMV(Real* rf,  Real* tf, Real* zf, Real* cf, const int* Nbin,
               ARLIM_P(alo), ARLIM_P(ahi));
        };


static
void
PrintUsage (const char* progName)
{
    std::cout << '\n';
    std::cout << "Usage:" << '\n';
    std::cout << progName << '\n';
    std::cout << "    infiles  = inputFileList" << '\n';
    std::cout << "   [help=<0,1>]" << '\n';
    std::cout << "   [bsize=<n>]" << '\n';
    std::cout << "   [-verbose]" << '\n';
    std::cout << '\n';
    exit(1);
}

int
main (int   argc,
      char* argv[])
{
    BoxLib::Initialize(argc,argv);

    if (argc == 1)
        PrintUsage(argv[0]);
    ParmParse pp;

    if (pp.contains("help"))
        PrintUsage(argv[0]);

    FArrayBox::setFormat(FABio::FAB_IEEE_32);
    //
    // Scan the arguments.
    //
    bool verbose = false;
    if (pp.contains("verbose"))
    {
        verbose = true;
        AmrData::SetVerbose(true);
    }
    const int nfile = pp.countval("infiles");
    if (nfile<=0) 
        BoxLib::Abort("You must specify `infiles'");

    std::vector<std::string> iFile(nfile);
    pp.queryarr("infiles",iFile);

    int bsize=5; pp.query("bsize",bsize);

    DataServices::SetBatchMode();
    FileType fileType(NEWPLT);    

    const int nComp = BL_SPACEDIM + 2;
    Array<string> names(nComp);
    names[0] = "xvel";
    names[1] = "yvel";
    names[2] = "zvel";
    names[3] = "density";
    names[4] = "tracer";

    DataServices* dataServices0 = new DataServices(iFile[0], fileType);
    AmrData& amrData0 = dataServices0->AmrDataRef();
    const Box box = amrData0.ProbDomain()[0];

    // Get a shifted domnlo vector to center domain
    Array<Real> domnlo(BL_SPACEDIM);
    for (int i=0; i<domnlo.size(); i++)
        domnlo[i] = -0.5*(amrData0.ProbLo()[i] + amrData0.ProbHi()[i]);

    delete dataServices0;

    FArrayBox avVR, avVT, avVZ, avC;

    {
        std::string fabFile("avVR.fab");
        std::cout << "Reading " << fabFile << std::endl;
        std::ifstream isf(fabFile.c_str(),std::ios::in|std::ios::binary);
        avVR.readFrom(isf);
        isf.close();
    }

    {
        std::string fabFile("avVT.fab");
        std::cout << "Reading " << fabFile << std::endl;
        std::ifstream isf(fabFile.c_str(),std::ios::in|std::ios::binary);
        avVT.readFrom(isf);
        isf.close();
    }

    {
        std::string fabFile("avVZ.fab");
        std::cout << "Reading " << fabFile << std::endl;
        std::ifstream isf(fabFile.c_str(),std::ios::in|std::ios::binary);
        avVZ.readFrom(isf);
        isf.close();
    }

    {
        std::string fabFile("avC.fab");
        std::cout << "Reading " << fabFile << std::endl;
        std::ifstream isf(fabFile.c_str(),std::ios::in|std::ios::binary);
        avC.readFrom(isf);
        isf.close();
    }

    const Box& vbox = avVR.box();
    const int maxbin = vbox.length(1) - 1;
    const int planes = vbox.length(0);

    FArrayBox vData(box,nComp);
    FArrayBox tmp(box,1);

    FArrayBox flVR(vbox,1), flVRi(vbox,1);
    FArrayBox flVT(vbox,1), flVTi(vbox,1);
    FArrayBox flVZ(vbox,1), flVZi(vbox,1);
    FArrayBox flC(vbox,1),  flCi(vbox,1);
    BaseFab<int> Nbin(vbox,1);

    flVR.setVal(0);
    flVT.setVal(0);
    flVZ.setVal(0);
    flC.setVal(0);
    for (int ifile=0; ifile<iFile.size(); ++ifile)
    {
        std::cout << "Processing infile= " << iFile[ifile] << std::endl;

        DataServices dataServices(iFile[ifile], fileType);

        if (!dataServices.AmrDataOk())
            ParallelDescriptor::EndParallel();
        
        AmrData& amrData = dataServices.AmrDataRef();
        
        std::cout << "    Reading data" << std::endl;

        for (int i=0; i<nComp; ++i)
        {
            amrData.FillVar(&tmp,box,0,names[i],0);
            vData.copy(tmp,box,0,box,i,1);
        }
        
        std::cout << "    Transforming data" << std::endl;

        // Compute sum of squares of fluctuating vels for each bin/plane
        TRANSV(vData.dataPtr(), ARLIM(vData.loVect()), ARLIM(vData.hiVect()),
               amrData.DxLevel()[0].dataPtr(), domnlo.dataPtr(),
               avVR.dataPtr(), avVT.dataPtr(), avVZ.dataPtr(), avC.dataPtr(),
               Nbin.dataPtr(),
               ARLIM(Nbin.loVect()), ARLIM(Nbin.hiVect()), &maxbin, &bsize,
               flVRi.dataPtr(), flVTi.dataPtr(), flVZi.dataPtr(),  flCi.dataPtr());

        flVR.plus(flVRi);
        flVT.plus(flVTi);
        flVZ.plus(flVZi);
        flC.plus(flCi);
    }

    // Number in each bin is the same over all files, set Nbin to total
    Nbin.mult(iFile.size());
    
    // Normalize 
    NORMV(flVR.dataPtr(), flVT.dataPtr(), flVZ.dataPtr(), flC.dataPtr(),
          Nbin.dataPtr(), ARLIM(Nbin.loVect()), ARLIM(Nbin.hiVect()));

    {
        std::string oFile("flVR.fab");
        std::cout << "Writing " << oFile << std::endl;
        std::ofstream osf(oFile.c_str(),std::ios::out|std::ios::binary);
        flVR.writeOn(osf);
        osf.close();
    }

    {
        std::string oFile("flVT.fab");
        std::cout << "Writing " << oFile << std::endl;
        std::ofstream osf(oFile.c_str(),std::ios::out|std::ios::binary);
        flVT.writeOn(osf);
        osf.close();
    }

    {
        std::string oFile("flVZ.fab");
        std::cout << "Writing " << oFile << std::endl;
        std::ofstream osf(oFile.c_str(),std::ios::out|std::ios::binary);
        flVZ.writeOn(osf);
        osf.close();
    }

    {
        std::string oFile("flC.fab");
        std::cout << "Writing " << oFile << std::endl;
        std::ofstream osf(oFile.c_str(),std::ios::out|std::ios::binary);
        flC.writeOn(osf);
        osf.close();
    }

    BoxLib::Finalize();
    return 0;
}
