/*
 * Copyright (c) 2014-2020 Pavel Kalvoda <me@pavelkalvoda.com>
 *
 * libcbor is free software; you can redistribute it and/or modify
 * it under the terms of the MIT license. See LICENSE for details.
 */

#include "assertions.h"
#include "cbor.h"
#include "test_allocator.h"

cbor_item_t *bs;
struct cbor_load_result res;

unsigned char data1[] = {0x40, 0xFF};
unsigned char data2[] = {0x41, 0xA1};
unsigned char data3[] = {0x58, 0x01, 0xA1, 0xFF};
unsigned char data4[] = {
    0x58, 0xFF, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09,
    0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15,
    0x16, 0x17, 0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F, 0x20, 0x21,
    0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2A, 0x2B, 0x2C, 0x2D,
    0x2E, 0x2F, 0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39,
    0x3A, 0x3B, 0x3C, 0x3D, 0x3E, 0x3F, 0x40, 0x41, 0x42, 0x43, 0x44, 0x45,
    0x46, 0x47, 0x48, 0x49, 0x4A, 0x4B, 0x4C, 0x4D, 0x4E, 0x4F, 0x50, 0x51,
    0x52, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58, 0x59, 0x5A, 0x5B, 0x5C, 0x5D,
    0x5E, 0x5F, 0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x69,
    0x6A, 0x6B, 0x6C, 0x6D, 0x6E, 0x6F, 0x70, 0x71, 0x72, 0x73, 0x74, 0x75,
    0x76, 0x77, 0x78, 0x79, 0x7A, 0x7B, 0x7C, 0x7D, 0x7E, 0x7F, 0x80, 0x81,
    0x82, 0x83, 0x84, 0x85, 0x86, 0x87, 0x88, 0x89, 0x8A, 0x8B, 0x8C, 0x8D,
    0x8E, 0x8F, 0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98, 0x99,
    0x9A, 0x9B, 0x9C, 0x9D, 0x9E, 0x9F, 0xA0, 0xA1, 0xA2, 0xA3, 0xA4, 0xA5,
    0xA6, 0xA7, 0xA8, 0xA9, 0xAA, 0xAB, 0xAC, 0xAD, 0xAE, 0xAF, 0xB0, 0xB1,
    0xB2, 0xB3, 0xB4, 0xB5, 0xB6, 0xB7, 0xB8, 0xB9, 0xBA, 0xBB, 0xBC, 0xBD,
    0xBE, 0xBF, 0xC0, 0xC1, 0xC2, 0xC3, 0xC4, 0xC5, 0xC6, 0xC7, 0xC8, 0xC9,
    0xCA, 0xCB, 0xCC, 0xCD, 0xCE, 0xCF, 0xD0, 0xD1, 0xD2, 0xD3, 0xD4, 0xD5,
    0xD6, 0xD7, 0xD8, 0xD9, 0xDA, 0xDB, 0xDC, 0xDD, 0xDE, 0xDF, 0xE0, 0xE1,
    0xE2, 0xE3, 0xE4, 0xE5, 0xE6, 0xE7, 0xE8, 0xE9, 0xEA, 0xEB, 0xEC, 0xED,
    0xEE, 0xEF, 0xF0, 0xF1, 0xF2, 0xF3, 0xF4, 0xF5, 0xF6, 0xF7, 0xF8, 0xF9,
    0xFA, 0xFB, 0xFC, 0xFD, 0xFE, 0xFF};
unsigned char data5[] = {
    0x59, 0x00, 0xFF, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
    0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F, 0x10, 0x11, 0x12, 0x13, 0x14,
    0x15, 0x16, 0x17, 0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F, 0x20,
    0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2A, 0x2B, 0x2C,
    0x2D, 0x2E, 0x2F, 0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38,
    0x39, 0x3A, 0x3B, 0x3C, 0x3D, 0x3E, 0x3F, 0x40, 0x41, 0x42, 0x43, 0x44,
    0x45, 0x46, 0x47, 0x48, 0x49, 0x4A, 0x4B, 0x4C, 0x4D, 0x4E, 0x4F, 0x50,
    0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58, 0x59, 0x5A, 0x5B, 0x5C,
    0x5D, 0x5E, 0x5F, 0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68,
    0x69, 0x6A, 0x6B, 0x6C, 0x6D, 0x6E, 0x6F, 0x70, 0x71, 0x72, 0x73, 0x74,
    0x75, 0x76, 0x77, 0x78, 0x79, 0x7A, 0x7B, 0x7C, 0x7D, 0x7E, 0x7F, 0x80,
    0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87, 0x88, 0x89, 0x8A, 0x8B, 0x8C,
    0x8D, 0x8E, 0x8F, 0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98,
    0x99, 0x9A, 0x9B, 0x9C, 0x9D, 0x9E, 0x9F, 0xA0, 0xA1, 0xA2, 0xA3, 0xA4,
    0xA5, 0xA6, 0xA7, 0xA8, 0xA9, 0xAA, 0xAB, 0xAC, 0xAD, 0xAE, 0xAF, 0xB0,
    0xB1, 0xB2, 0xB3, 0xB4, 0xB5, 0xB6, 0xB7, 0xB8, 0xB9, 0xBA, 0xBB, 0xBC,
    0xBD, 0xBE, 0xBF, 0xC0, 0xC1, 0xC2, 0xC3, 0xC4, 0xC5, 0xC6, 0xC7, 0xC8,
    0xC9, 0xCA, 0xCB, 0xCC, 0xCD, 0xCE, 0xCF, 0xD0, 0xD1, 0xD2, 0xD3, 0xD4,
    0xD5, 0xD6, 0xD7, 0xD8, 0xD9, 0xDA, 0xDB, 0xDC, 0xDD, 0xDE, 0xDF, 0xE0,
    0xE1, 0xE2, 0xE3, 0xE4, 0xE5, 0xE6, 0xE7, 0xE8, 0xE9, 0xEA, 0xEB, 0xEC,
    0xED, 0xEE, 0xEF, 0xF0, 0xF1, 0xF2, 0xF3, 0xF4, 0xF5, 0xF6, 0xF7, 0xF8,
    0xF9, 0xFA, 0xFB, 0xFC, 0xFD, 0xFE, 0xFF};
unsigned char data6[] = {
    0x5A, 0x00, 0x00, 0x00, 0xFF, 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06,
    0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F, 0x10, 0x11, 0x12,
    0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E,
    0x1F, 0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2A,
    0x2B, 0x2C, 0x2D, 0x2E, 0x2F, 0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36,
    0x37, 0x38, 0x39, 0x3A, 0x3B, 0x3C, 0x3D, 0x3E, 0x3F, 0x40, 0x41, 0x42,
    0x43, 0x44, 0x45, 0x46, 0x47, 0x48, 0x49, 0x4A, 0x4B, 0x4C, 0x4D, 0x4E,
    0x4F, 0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57, 0x58, 0x59, 0x5A,
    0x5B, 0x5C, 0x5D, 0x5E, 0x5F, 0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66,
    0x67, 0x68, 0x69, 0x6A, 0x6B, 0x6C, 0x6D, 0x6E, 0x6F, 0x70, 0x71, 0x72,
    0x73, 0x74, 0x75, 0x76, 0x77, 0x78, 0x79, 0x7A, 0x7B, 0x7C, 0x7D, 0x7E,
    0x7F, 0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87, 0x88, 0x89, 0x8A,
    0x8B, 0x8C, 0x8D, 0x8E, 0x8F, 0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96,
    0x97, 0x98, 0x99, 0x9A, 0x9B, 0x9C, 0x9D, 0x9E, 0x9F, 0xA0, 0xA1, 0xA2,
    0xA3, 0xA4, 0xA5, 0xA6, 0xA7, 0xA8, 0xA9, 0xAA, 0xAB, 0xAC, 0xAD, 0xAE,
    0xAF, 0xB0, 0xB1, 0xB2, 0xB3, 0xB4, 0xB5, 0xB6, 0xB7, 0xB8, 0xB9, 0xBA,
    0xBB, 0xBC, 0xBD, 0xBE, 0xBF, 0xC0, 0xC1, 0xC2, 0xC3, 0xC4, 0xC5, 0xC6,
    0xC7, 0xC8, 0xC9, 0xCA, 0xCB, 0xCC, 0xCD, 0xCE, 0xCF, 0xD0, 0xD1, 0xD2,
    0xD3, 0xD4, 0xD5, 0xD6, 0xD7, 0xD8, 0xD9, 0xDA, 0xDB, 0xDC, 0xDD, 0xDE,
    0xDF, 0xE0, 0xE1, 0xE2, 0xE3, 0xE4, 0xE5, 0xE6, 0xE7, 0xE8, 0xE9, 0xEA,
    0xEB, 0xEC, 0xED, 0xEE, 0xEF, 0xF0, 0xF1, 0xF2, 0xF3, 0xF4, 0xF5, 0xF6,
    0xF7, 0xF8, 0xF9, 0xFA, 0xFB, 0xFC, 0xFD, 0xFE, 0xFF};
unsigned char data7[] = {
    0x5B, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF, 0x00, 0x01, 0x02,
    0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E,
    0x0F, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19, 0x1A,
    0x1B, 0x1C, 0x1D, 0x1E, 0x1F, 0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26,
    0x27, 0x28, 0x29, 0x2A, 0x2B, 0x2C, 0x2D, 0x2E, 0x2F, 0x30, 0x31, 0x32,
    0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3A, 0x3B, 0x3C, 0x3D, 0x3E,
    0x3F, 0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48, 0x49, 0x4A,
    0x4B, 0x4C, 0x4D, 0x4E, 0x4F, 0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56,
    0x57, 0x58, 0x59, 0x5A, 0x5B, 0x5C, 0x5D, 0x5E, 0x5F, 0x60, 0x61, 0x62,
    0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x69, 0x6A, 0x6B, 0x6C, 0x6D, 0x6E,
    0x6F, 0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77, 0x78, 0x79, 0x7A,
    0x7B, 0x7C, 0x7D, 0x7E, 0x7F, 0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86,
    0x87, 0x88, 0x89, 0x8A, 0x8B, 0x8C, 0x8D, 0x8E, 0x8F, 0x90, 0x91, 0x92,
    0x93, 0x94, 0x95, 0x96, 0x97, 0x98, 0x99, 0x9A, 0x9B, 0x9C, 0x9D, 0x9E,
    0x9F, 0xA0, 0xA1, 0xA2, 0xA3, 0xA4, 0xA5, 0xA6, 0xA7, 0xA8, 0xA9, 0xAA,
    0xAB, 0xAC, 0xAD, 0xAE, 0xAF, 0xB0, 0xB1, 0xB2, 0xB3, 0xB4, 0xB5, 0xB6,
    0xB7, 0xB8, 0xB9, 0xBA, 0xBB, 0xBC, 0xBD, 0xBE, 0xBF, 0xC0, 0xC1, 0xC2,
    0xC3, 0xC4, 0xC5, 0xC6, 0xC7, 0xC8, 0xC9, 0xCA, 0xCB, 0xCC, 0xCD, 0xCE,
    0xCF, 0xD0, 0xD1, 0xD2, 0xD3, 0xD4, 0xD5, 0xD6, 0xD7, 0xD8, 0xD9, 0xDA,
    0xDB, 0xDC, 0xDD, 0xDE, 0xDF, 0xE0, 0xE1, 0xE2, 0xE3, 0xE4, 0xE5, 0xE6,
    0xE7, 0xE8, 0xE9, 0xEA, 0xEB, 0xEC, 0xED, 0xEE, 0xEF, 0xF0, 0xF1, 0xF2,
    0xF3, 0xF4, 0xF5, 0xF6, 0xF7, 0xF8, 0xF9, 0xFA, 0xFB, 0xFC, 0xFD, 0xFE,
    0xFF};
unsigned char data8[] = {
    0x5C, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x00, 0x01, 0xFF, 0x00, 0x01,
    0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D,
    0x0E, 0x0F, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18, 0x19,
    0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F, 0x20, 0x21, 0x22, 0x23, 0x24, 0x25,
    0x26, 0x27, 0x28, 0x29, 0x2A, 0x2B, 0x2C, 0x2D, 0x2E, 0x2F, 0x30, 0x31,
    0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3A, 0x3B, 0x3C, 0x3D,
    0x3E, 0x3F, 0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48, 0x49,
    0x4A, 0x4B, 0x4C, 0x4D, 0x4E, 0x4F, 0x50, 0x51, 0x52, 0x53, 0x54, 0x55,
    0x56, 0x57, 0x58, 0x59, 0x5A, 0x5B, 0x5C, 0x5D, 0x5E, 0x5F, 0x60, 0x61,
    0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x69, 0x6A, 0x6B, 0x6C, 0x6D,
    0x6E, 0x6F, 0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77, 0x78, 0x79,
    0x7A, 0x7B, 0x7C, 0x7D, 0x7E, 0x7F, 0x80, 0x81, 0x82, 0x83, 0x84, 0x85,
    0x86, 0x87, 0x88, 0x89, 0x8A, 0x8B, 0x8C, 0x8D, 0x8E, 0x8F, 0x90, 0x91,
    0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98, 0x99, 0x9A, 0x9B, 0x9C, 0x9D,
    0x9E, 0x9F, 0xA0, 0xA1, 0xA2, 0xA3, 0xA4, 0xA5, 0xA6, 0xA7, 0xA8, 0xA9,
    0xAA, 0xAB, 0xAC, 0xAD, 0xAE, 0xAF, 0xB0, 0xB1, 0xB2, 0xB3, 0xB4, 0xB5,
    0xB6, 0xB7, 0xB8, 0xB9, 0xBA, 0xBB, 0xBC, 0xBD, 0xBE, 0xBF, 0xC0, 0xC1,
    0xC2, 0xC3, 0xC4, 0xC5, 0xC6, 0xC7, 0xC8, 0xC9, 0xCA, 0xCB, 0xCC, 0xCD,
    0xCE, 0xCF, 0xD0, 0xD1, 0xD2, 0xD3, 0xD4, 0xD5, 0xD6, 0xD7, 0xD8, 0xD9,
    0xDA, 0xDB, 0xDC, 0xDD, 0xDE, 0xDF, 0xE0, 0xE1, 0xE2, 0xE3, 0xE4, 0xE5,
    0xE6, 0xE7, 0xE8, 0xE9, 0xEA, 0xEB, 0xEC, 0xED, 0xEE, 0xEF, 0xF0, 0xF1,
    0xF2, 0xF3, 0xF4, 0xF5, 0xF6, 0xF7, 0xF8, 0xF9, 0xFA, 0xFB, 0xFC, 0xFD,
    0xFE, 0xFF};

static void test_empty_bs(void **_CBOR_UNUSED(_state)) {
  bs = cbor_load(data1, 2, &res);
  assert_non_null(bs);
  assert_true(cbor_typeof(bs) == CBOR_TYPE_BYTESTRING);
  assert_true(cbor_isa_bytestring(bs));
  assert_size_equal(cbor_bytestring_length(bs), 0);
  assert_true(res.read == 1);
  cbor_decref(&bs);
  assert_null(bs);
}

static void test_embedded_bs(void **_CBOR_UNUSED(_state)) {
  bs = cbor_load(data2, 2, &res);
  assert_non_null(bs);
  assert_true(cbor_typeof(bs) == CBOR_TYPE_BYTESTRING);
  assert_true(cbor_isa_bytestring(bs));
  assert_true(cbor_bytestring_length(bs) == 1);
  assert_true(res.read == 2);
  assert_true(*cbor_bytestring_handle(bs) == 0xA1);
  assert_memory_equal(cbor_bytestring_handle(bs), data2 + 1, 1);
  cbor_decref(&bs);
  assert_null(bs);
}

static void test_notenough_data(void **_CBOR_UNUSED(_state)) {
  bs = cbor_load(data3, 2, &res);
  assert_null(bs);
  assert_true(res.error.code == CBOR_ERR_NOTENOUGHDATA);
}

static void test_short_bs1(void **_CBOR_UNUSED(_state)) {
  bs = cbor_load(data3, 4, &res);
  assert_non_null(bs);
  assert_true(cbor_typeof(bs) == CBOR_TYPE_BYTESTRING);
  assert_true(cbor_isa_bytestring(bs));
  assert_true(cbor_bytestring_length(bs) == 1);
  assert_true(res.read == 3);
  assert_true(*cbor_bytestring_handle(bs) == 0xA1);
  assert_memory_equal(cbor_bytestring_handle(bs), data3 + 2, 1);
  cbor_decref(&bs);
  assert_null(bs);
}

static void test_short_bs2(void **_CBOR_UNUSED(_state)) {
  bs = cbor_load(data4, 259, &res);
  assert_non_null(bs);
  assert_true(cbor_typeof(bs) == CBOR_TYPE_BYTESTRING);
  assert_true(cbor_isa_bytestring(bs));
  assert_true(cbor_bytestring_length(bs) == 255);
  assert_true(res.read == 257);
  assert_memory_equal(cbor_bytestring_handle(bs), data4 + 2, 0xFF);
  cbor_decref(&bs);
  assert_null(bs);
}

static void test_half_bs(void **_CBOR_UNUSED(_state)) {
  bs = cbor_load(data5, 259, &res);
  assert_non_null(bs);
  assert_true(cbor_typeof(bs) == CBOR_TYPE_BYTESTRING);
  assert_true(cbor_isa_bytestring(bs));
  assert_true(cbor_bytestring_length(bs) == 255);
  assert_true(res.read == 258);
  assert_memory_equal(cbor_bytestring_handle(bs), data5 + 3, 0xFF);
  cbor_decref(&bs);
  assert_null(bs);
}

static void test_int_bs(void **_CBOR_UNUSED(_state)) {
  bs = cbor_load(data6, 261, &res);
  assert_non_null(bs);
  assert_true(cbor_typeof(bs) == CBOR_TYPE_BYTESTRING);
  assert_true(cbor_isa_bytestring(bs));
  assert_true(cbor_bytestring_length(bs) == 255);
  assert_true(res.read == 260);
  assert_memory_equal(cbor_bytestring_handle(bs), data6 + 5, 0xFF);
  cbor_decref(&bs);
  assert_null(bs);
}

static void test_long_bs(void **_CBOR_UNUSED(_state)) {
  bs = cbor_load(data7, 265, &res);
  assert_non_null(bs);
  assert_true(cbor_typeof(bs) == CBOR_TYPE_BYTESTRING);
  assert_true(cbor_isa_bytestring(bs));
  assert_true(cbor_bytestring_length(bs) == 255);
  assert_true(res.read == 264);
  assert_memory_equal(cbor_bytestring_handle(bs), data7 + 9, 0xFF);
  cbor_decref(&bs);
  assert_null(bs);
}

unsigned char data9[] = {0x5F, 0xFF};

static void test_zero_indef(void **_CBOR_UNUSED(_state)) {
  bs = cbor_load(data9, 2, &res);
  assert_non_null(bs);
  assert_true(cbor_typeof(bs) == CBOR_TYPE_BYTESTRING);
  assert_true(cbor_isa_bytestring(bs));
  assert_true(cbor_bytestring_is_indefinite(bs));
  assert_true(cbor_bytestring_chunk_count(bs) == 0);
  assert_true(res.read == 2);
  cbor_decref(&bs);
  assert_null(bs);
}

unsigned char data10[] = {0x5F, 0x58, 0x01, 0xA1, 0xFF, 0xFF};

/*                          start |   bstring     | break| extra */

static void test_short_indef(void **_CBOR_UNUSED(_state)) {
  bs = cbor_load(data10, 6, &res);
  assert_non_null(bs);
  assert_true(cbor_typeof(bs) == CBOR_TYPE_BYTESTRING);
  assert_true(cbor_isa_bytestring(bs));
  assert_true(cbor_bytestring_length(bs) == 0);
  assert_true(cbor_bytestring_is_indefinite(bs));
  assert_true(cbor_bytestring_chunk_count(bs) == 1);
  assert_true(res.read == 5);
  assert_true(cbor_isa_bytestring(cbor_bytestring_chunks_handle(bs)[0]));
  assert_true(cbor_bytestring_length(cbor_bytestring_chunks_handle(bs)[0]) ==
              1);
  assert_true(*cbor_bytestring_handle(cbor_bytestring_chunks_handle(bs)[0]) ==
              0xA1);
  cbor_decref(&bs);
  assert_null(bs);
}

unsigned char data11[] = {0x5F, 0x58, 0x01, 0xA1, 0x58, 0x01, 0xA2, 0xFF, 0xFF};

/*                          start |   bstring     |    bstring     | break|
 * extra */

static void test_two_indef(void **_CBOR_UNUSED(_state)) {
  bs = cbor_load(data11, 9, &res);
  assert_non_null(bs);
  assert_size_equal(1, cbor_refcount(bs));
  assert_true(cbor_typeof(bs) == CBOR_TYPE_BYTESTRING);
  assert_true(cbor_isa_bytestring(bs));
  assert_true(cbor_bytestring_length(bs) == 0);
  assert_true(cbor_bytestring_is_indefinite(bs));
  assert_true(cbor_bytestring_chunk_count(bs) == 2);
  assert_true(res.read == 8);
  assert_true(cbor_isa_bytestring(cbor_bytestring_chunks_handle(bs)[0]));
  assert_true(cbor_bytestring_length(cbor_bytestring_chunks_handle(bs)[0]) ==
              1);
  assert_true(*cbor_bytestring_handle(cbor_bytestring_chunks_handle(bs)[0]) ==
              0xA1);
  assert_true(cbor_isa_bytestring(cbor_bytestring_chunks_handle(bs)[1]));
  assert_true(cbor_bytestring_length(cbor_bytestring_chunks_handle(bs)[1]) ==
              1);
  assert_true(*cbor_bytestring_handle(cbor_bytestring_chunks_handle(bs)[1]) ==
              0xA2);
  cbor_decref(&bs);
  assert_null(bs);
}

unsigned char data12[] = {0x5F, 0x58, 0x01};

/*                          start |   bstring - too short */

static void test_missing_indef(void **_CBOR_UNUSED(_state)) {
  bs = cbor_load(data12, 3, &res);
  assert_true(res.error.code == CBOR_ERR_NOTENOUGHDATA);
  assert_null(bs);
}

static void test_inline_creation(void **_CBOR_UNUSED(_state)) {
  bs = cbor_build_bytestring((cbor_data) "Hello!", 6);
  assert_memory_equal(cbor_bytestring_handle(bs), "Hello!", 6);
  cbor_decref(&bs);
}

static void test_add_chunk_reallocation_overflow(void **_CBOR_UNUSED(_state)) {
  bs = cbor_new_indefinite_bytestring();
  cbor_item_t *chunk = cbor_build_bytestring((cbor_data) "Hello!", 6);
  struct cbor_indefinite_string_data *metadata =
      (struct cbor_indefinite_string_data *)bs->data;
  // Pretend we already have many chunks allocated
  metadata->chunk_count = SIZE_MAX;
  metadata->chunk_capacity = SIZE_MAX;

  assert_false(cbor_bytestring_add_chunk(bs, chunk));
  assert_size_equal(cbor_refcount(chunk), 1);

  metadata->chunk_count = 0;
  metadata->chunk_capacity = 0;
  cbor_decref(&chunk);
  cbor_decref(&bs);
}

static void test_bytestring_creation(void **_CBOR_UNUSED(_state)) {
  WITH_FAILING_MALLOC({ assert_null(cbor_new_definite_bytestring()); });

  WITH_FAILING_MALLOC({ assert_null(cbor_new_indefinite_bytestring()); });
  WITH_MOCK_MALLOC({ assert_null(cbor_new_indefinite_bytestring()); }, 2,
                   MALLOC, MALLOC_FAIL);

  unsigned char bytes[] = {0, 0, 0xFF, 0xAB};

  WITH_FAILING_MALLOC({ assert_null(cbor_build_bytestring(bytes, 4)); });
  WITH_MOCK_MALLOC({ assert_null(cbor_build_bytestring(bytes, 4)); }, 2, MALLOC,
                   MALLOC_FAIL);
}

static void test_bytestring_add_chunk(void **_CBOR_UNUSED(_state)) {
  unsigned char bytes[] = {0, 0, 0xFF, 0xAB};
  WITH_MOCK_MALLOC(
      {
        cbor_item_t *bytestring = cbor_new_indefinite_bytestring();
        cbor_item_t *chunk = cbor_build_bytestring(bytes, 4);

        assert_false(cbor_bytestring_add_chunk(bytestring, chunk));

        assert_size_equal(cbor_bytestring_chunk_count(bytestring), 0);
        assert_size_equal(
            ((struct cbor_indefinite_string_data *)bytestring->data)
                ->chunk_capacity,
            0);

        cbor_decref(&chunk);
        cbor_decref(&bytestring);
      },
      5, MALLOC, MALLOC, MALLOC, MALLOC, REALLOC_FAIL);
}

int main(void) {
  const struct CMUnitTest tests[] = {
      cmocka_unit_test(test_empty_bs),
      cmocka_unit_test(test_embedded_bs),
      cmocka_unit_test(test_notenough_data),
      cmocka_unit_test(test_short_bs1),
      cmocka_unit_test(test_short_bs2),
      cmocka_unit_test(test_half_bs),
      cmocka_unit_test(test_int_bs),
      cmocka_unit_test(test_long_bs),
      cmocka_unit_test(test_zero_indef),
      cmocka_unit_test(test_short_indef),
      cmocka_unit_test(test_two_indef),
      cmocka_unit_test(test_missing_indef),
      cmocka_unit_test(test_inline_creation),
      cmocka_unit_test(test_add_chunk_reallocation_overflow),
      cmocka_unit_test(test_bytestring_creation),
      cmocka_unit_test(test_bytestring_add_chunk),
  };
  return cmocka_run_group_tests(tests, NULL, NULL);
}
