/*
 * This program is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License, version 2.1 as published by the Free Software
 * Foundation.
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * program; if not, you can obtain a copy at http://www.gnu.org/licenses/old-licenses/lgpl-2.1.html
 * or from the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 *
 * Copyright (c) 2006 - 2009 Pentaho Corporation and Contributors.  All rights reserved.
 */

package org.pentaho.reporting.libraries.fonts.encoding;

/**
 * This is an autogenerated file. The charset contents of this file have been
 * read from '${source}'. This class supports the encoding '${encoding}'.
 */
public final class External8BitEncodingCore implements EncodingCore
{
  private char[] byteToChar;
  private byte[] charToByte;

  public External8BitEncodingCore(final External8BitEncodingData resource)
  {
    if (resource == null)
    {
      throw new NullPointerException();
    }

    final int[] indexDelta = resource.getIndexDelta();
    final int[] valueDelta = resource.getValueDelta();

    // decompress ...
    byteToChar = new char[256];
    final int encodingLength = byteToChar.length;
    for (int i = 0; i < encodingLength; i++)
    {
      byteToChar[i] = (char) i;
    }

    int index = 0;
    int value = 0;
    final int indexDeltaLength = indexDelta.length;
    for (int i = 0; i < indexDeltaLength; i++)
    {
      index += indexDelta[i];
      value += valueDelta[i];
      byteToChar[index] = (char) value;
    }

    // build the backward reference list ...
    charToByte = new byte[65536];
    for (int i = 0; i < encodingLength; i++)
    {
      charToByte[byteToChar[i]] = (byte) i;
    }
  }

  public boolean isUnicodeCharacterSupported(final int c)
  {
    if (c == 0)
    {
      return true;
    }
    return (charToByte[c] != 0);
  }

  /**
   * Encode, but ignore errors.
   *
   * @param text
   * @param buffer
   * @return
   */
  public ByteBuffer encode(final CodePointBuffer text, ByteBuffer buffer)
  {
    final int textLength = text.getLength();
    if (buffer == null)
    {
      buffer = new ByteBuffer(textLength);
    }
    else if (buffer.getLength() < textLength)
    {
      buffer.ensureSize(textLength);
    }

    final byte[] targetArray = buffer.getData();
    final int[] sourceArray = text.getData();

    int targetIdx = buffer.getOffset();
    final int endPos = text.getCursor();
    for (int i = text.getOffset(); i < endPos; i++)
    {
      final int sourceItem = sourceArray[i];
      if (isUnicodeCharacterSupported(sourceItem))
      {
        targetArray[targetIdx] = charToByte[(sourceItem & 0xffff)];
        targetIdx += 1;
      }
    }

    buffer.setCursor(targetIdx);
    return buffer;
  }

  public ByteBuffer encode(final CodePointBuffer text,
                           ByteBuffer buffer,
                           final EncodingErrorType errorHandling)
          throws EncodingException
  {
    final int textLength = text.getLength();
    if (buffer == null)
    {
      buffer = new ByteBuffer(textLength);
    }
    else if (buffer.getLength() < textLength)
    {
      buffer.ensureSize(textLength);
    }

    final byte[] targetArray = buffer.getData();
    final int[] sourceArray = text.getData();

    int targetIdx = buffer.getOffset();
    final int endPos = text.getCursor();
    for (int i = text.getOffset(); i < endPos; i++)
    {
      final int sourceItem = sourceArray[i];
      if (isUnicodeCharacterSupported(sourceItem))
      {
        targetArray[targetIdx] = charToByte[(sourceItem & 0xffff)];
        targetIdx += 1;
      }
      else
      {
        if (EncodingErrorType.REPLACE.equals(errorHandling))
        {
          targetArray[targetIdx] = (byte) ('?' & 0xff);
          targetIdx += 1;
        }
        else if (EncodingErrorType.FAIL.equals(errorHandling))
        {
          throw new EncodingException();
        }
      }
    }

    buffer.setCursor(targetIdx);
    return buffer;
  }

  public CodePointBuffer decode(final ByteBuffer text, CodePointBuffer buffer)
  {
    final int textLength = text.getLength();
    if (buffer == null)
    {
      buffer = new CodePointBuffer(textLength);
    }
    else if (buffer.getLength() < textLength)
    {
      buffer.ensureSize(textLength);
    }

    final int[] targetArray = buffer.getData();
    final byte[] sourceArray = text.getData();

    int targetIdx = buffer.getOffset();
    final int endPos = text.getCursor();
    for (int i = text.getOffset(); i < endPos; i++)
    {
      targetArray[targetIdx] = byteToChar[(sourceArray[i] & 0xff)];
      targetIdx += 1;
    }

    buffer.setCursor(targetIdx);
    return buffer;
  }

  public CodePointBuffer decode(final ByteBuffer text,
                                CodePointBuffer buffer,
                                final EncodingErrorType errorHandling)
  {
    final int textLength = text.getLength();
    if (buffer == null)
    {
      buffer = new CodePointBuffer(textLength);
    }
    else if (buffer.getLength() < textLength)
    {
      buffer.ensureSize(textLength);
    }

    final int[] targetArray = buffer.getData();
    final byte[] sourceArray = text.getData();

    int targetIdx = buffer.getOffset();
    final int endPos = text.getCursor();
    for (int i = text.getOffset(); i < endPos; i++)
    {
      targetArray[targetIdx] = byteToChar[(sourceArray[i] & 0xff)];
      targetIdx += 1;
    }

    buffer.setCursor(targetIdx);
    return buffer;
  }
}
