/*
 * YICS: Connect a FICS interface to the Yahoo! Chess server.
 * Copyright (C) 2004  Chris Howie
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; either version 2 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "types.h"
#include "vars.h"
#include "sockets.h"
#include "console.h"
#include "network.h"
#include "ropcodes.h"
#include "topcodes.h"
#include "util.h"
#include "command.h"
#include "globals.h"
#include "style.h"
#include "formula.h"

bool var_formula(Variable *, void *);
bool var_interface(Variable *, void *);
bool var_style(Variable *, void *);
bool var_tolerance(Variable *, void *);

bool tvar_timeinc(Variable *, void *);

void varset_autostart(Variable *);
void varset_ivar(Variable *);
void varset_lock(Variable *);
void varset_notimeout(Variable *);
void varset_open(Variable *);
void varset_pin(Variable *);
void varset_ptime(Variable *);
void varset_seek(Variable *);
void varset_shout(Variable *);
void varset_singleboard(Variable *);
void varset_style(Variable *);
void varset_timeinc(Variable *);
void varset_tolerance(Variable *);

void varset_formula(Variable *);

void tvarset_timeinc(Variable *);
void tvarset_rated(Variable *);
void tvarset_protection(Variable *);

Variable variables[] = {
/*	 name		validator	post-set	type		locks	string	number */
	{"autostart",	NULL,		varset_autostart, VAR_BOOLEAN,	false,	NULL,	0},
	{"inc",		tvar_timeinc,	varset_timeinc,	VAR_NUMERIC,	false,	NULL,	0},
	{"lock",	NULL,		varset_lock,	VAR_BOOLEAN,	true,	NULL,	0},
	{"ms",		NULL,		varset_ivar,	VAR_BOOLEAN,	true,	NULL,	0},
	{"notimeout",	NULL,		varset_notimeout, VAR_BOOLEAN,	false,	NULL,	0},
	{"open",	NULL,		varset_open,	VAR_BOOLEAN,	false,	NULL,	1},
	{"pin",		NULL,		varset_pin,	VAR_BOOLEAN,	false,	NULL,	0},
	{"ptime",	NULL,		varset_ptime,	VAR_BOOLEAN,	false,	NULL,	0},
	{"rated",	NULL,		varset_timeinc,	VAR_BOOLEAN,	false,	NULL,	1},
	{"seek",	NULL,		varset_seek,	VAR_BOOLEAN,	false,	NULL,	0},
	{"shout",	NULL,		varset_shout,	VAR_BOOLEAN,	false,	NULL,	1},
	{"singleboard",	NULL,		varset_singleboard, VAR_BOOLEAN,true,	NULL,	1},
	{"style",	var_style,	varset_style,	VAR_NUMERIC,	false,	NULL,	12},
	{"time",	tvar_timeinc,	varset_timeinc,	VAR_NUMERIC,	false,	NULL,	5},
	{"tolerance",	var_tolerance,	varset_tolerance, VAR_NUMERIC,	false,	NULL,	2},

	{"interface",	var_interface,	varset_lock,	VAR_STRING,	true,	NULL,	0},
	{"formula",	validateFormula,varset_formula,	VAR_STRING,	false,	NULL,	0},
	{"f1",		validateFormula,varset_formula,	VAR_STRING,	false,	NULL,	1},
	{"f2",		validateFormula,varset_formula,	VAR_STRING,	false,	NULL,	2},
	{"f3",		validateFormula,varset_formula,	VAR_STRING,	false,	NULL,	3},
	{"f4",		validateFormula,varset_formula,	VAR_STRING,	false,	NULL,	4},
	{"f5",		validateFormula,varset_formula,	VAR_STRING,	false,	NULL,	5},
	{"f6",		validateFormula,varset_formula,	VAR_STRING,	false,	NULL,	6},
	{"f7",		validateFormula,varset_formula,	VAR_STRING,	false,	NULL,	7},
	{"f8",		validateFormula,varset_formula,	VAR_STRING,	false,	NULL,	8},
	{"f9",		validateFormula,varset_formula,	VAR_STRING,	false,	NULL,	9},
	{"",		NULL,		NULL,		VAR_END,	false,	NULL,	0},
};

Variable tvariables[] = {
/*	 name		validator	post-set	type		locks	string	number */
	{"increment",	tvar_timeinc,	tvarset_timeinc,VAR_NUMERIC,	false,	NULL,	0},
	{"rated",	NULL,		tvarset_rated,	VAR_BOOLEAN,	false,	NULL,	0},
	{"protection",	var_tolerance,	tvarset_protection, VAR_NUMERIC,false,	NULL,	0},
	{"tenminlimit",	NULL,		tvarset_timeinc,VAR_BOOLEAN,	false,	NULL,	0},
	{"time",	tvar_timeinc,	tvarset_timeinc,VAR_NUMERIC,	false,	NULL,	0},
	{"",		NULL,		NULL,		VAR_END,	false,	NULL,	0},
};

static bool locked = false;

#if 0
int getvari(Variable *srch, const char *name) {
	while (srch->type != VAR_END) {
		if (!strcmp(name, srch->name))
			return srch->number;
		srch++;
	}

	return 0;
}

String *getvars(Variable *srch, const char *name) {
	while (srch->type != VAR_END) {
		if (!strcmp(name, srch->name))
			return srch->string;
		srch++;
	}

	return NULL;
}
#endif

int setvar(Variable *srch, const char *name, const char *setting, Variable **out) {
	String *var, *new = NULL;
	Variable *match = NULL;
	int len, num = -1, result = VARSET_OK;
	bool youlose = false;

	if (setting != NULL)
		new = StringNew(setting, -1);

	if (out != NULL)
		*out = NULL;

	var = StringNew(name, -1);
	lowercase(var->string);
	len = var->length;
	for (; srch->type != VAR_END; srch++) {
		if (!strcmp(var->string, srch->name)) {
			match = srch;
			break;
		} else if ((len <= (int)strlen(srch->name)) &&
		!memcmp(srch->name, var->string, len)) {
			if (match == NULL) {
				match = srch;
			} else {
				result = VARSET_AMBIGUOUS;
				goto VARSET_END;
			}
		}
	}

	if (match == NULL) {
		result = VARSET_NOTFOUND;
		goto VARSET_END;
	}

	if (out != NULL)
		*out = match;

	if (locked && match->locks) {
		result = VARSET_LOCKED;
		goto VARSET_END;
	}

	switch ((int)match->type) {
		case VAR_BOOLEAN:
			if (new == NULL) {
				num = !match->number;
			} else if (
			!strcmp(new->string, "1") ||
			!strcmp(new->string, "y") ||
			!strcmp(new->string, "yes") ||
			!strcmp(new->string, "on")) {
				num = 1;
			} else if (
			!strcmp(new->string, "0") ||
			!strcmp(new->string, "n") ||
			!strcmp(new->string, "no") ||
			!strcmp(new->string, "off")) {
				num = 0;
			} else {
				youlose = true;
			}
			break;

		case VAR_NUMERIC:
			if ((new != NULL) && isnumeric(new->string)) {
				num = atoi(new->string);
				youlose = (bool)!match->validator(match, (void *)&num);
			} else {
				youlose = true;
			}
			break;

		case VAR_STRING:
			if (new == NULL)
				new = StringNull();
			youlose = (bool)!match->validator(match, (void *)new->string);
	}

	if (youlose) {
		result = VARSET_BAD;
		goto VARSET_END;
	}

	switch ((int)match->type) {
		case VAR_BOOLEAN:
		case VAR_NUMERIC:
			match->number = num;
			break;

		case VAR_STRING:
			if (match->string != NULL)
				StringCopy(match->string, new);
			else
				match->string = StringCopy(StringNull(), new);
	}

	if (match->postset != NULL) {
		match->postset(match);
	} else {
		if (match->type == VAR_STRING)
			iprintf("Variable %s set to %s.\n",
				match->name, new->string);
		else
			iprintf("Variable %s set to %d.\n",
				match->name, num);

		prompt();
	}

/*
 * I know goto is ugly, but it's got to be WAY more maintainable than
 * triplicating clean-up code.
 */
VARSET_END:
	if (new != NULL)
		StringFree(new);
	StringFree(var);

	return result;
}

#define intp(x) (*((int *)(x)))

bool var_interface(Variable *var, void *new) {
	trim((char *)new);
	return true;
}

bool var_style(Variable *var, void *new) {
	return (bool)(getstyle(intp(new)) != NULL);
}

bool var_tolerance(Variable *var, void *new) {
	return (bool)(intp(new) <= 2);
}

bool tvar_timeinc(Variable *var, void *new) {
	return (bool)(intp(new) <= 999);
}

#undef intp

void varset_autostart(Variable *var) {
	iprintf("Games will no%c start automatically.\n",
		var->number ? 'w' : 't');
	prompt();
}

void varset_formula(Variable *var) {
	char *isset = ((var->string != NULL) &&
			(var->string->string[0] != '\0')) ? "" : "un";

	if (var->number == 0)
		iprintf("Formula %sset.\n", isset);
	else
		iprintf("Formula variable %s %sset.\n", var->name, isset);

	prompt();
}

void varset_ivar(Variable *var) {
	iprintf("%s %sset.\n", var->name, (var->number || var->string) ? "" : "un");
	prompt();
}

void varset_lock(Variable *var) {
	varset_ivar(var);
	locked = (bool)(var->number ? true : false);
}

void varset_notimeout(Variable *var) {
	iprintf("Timeout disconnection will no%c be prevented.\n",
		var->number ? 'w' : 't');
	prompt();
}

void varset_open(Variable *var) {
	char packet;
	String *decline[2];

	if (!var->number) {
		prompting = 0;	/* Prevent prompt flooding. */
		decline[1] = StringNew("I am going closed for invitations.", -1);
		while (invitations != NULL) {
			decline[0] = StringNew(invitations->who->lhandle, -1);
			com_decline(decline, 2);
			StringFree(decline[0]);
		}
		StringFree(decline[1]);
		prompting = 1;
	}

	iprintf("You are %s match requests.\n",
		var->number ? "now open to receive" : "no longer receiving");
	prompt();

	packet = (char)(var->number ? 0 : 1);
	nprintrop(ROP_OPEN, &packet, 1);
}

void varset_ptime(Variable *var) {
	iprintf("Your prompt will now%s show the time.\n",
		var->number ? "" : " not");
	prompt();
}

void varset_pin(Variable *var) {
	iprintf("You will no%c hear hear logins/logouts.\n",
		var->number ? 'w' : 't');
	prompt();
}

void varset_seek(Variable *var) {
	iprintf("You will no%c see seek ads.\n",
		var->number ? 'w' : 't');
	prompt();
}

void varset_shout(Variable *var) {
	iprintf("You will %s hear shouts.\n",
		var->number ? "now" : "no longer");
	prompt();
}

void varset_singleboard(Variable *var) {
	iprintf("singleboard %sset.\n",
		var->number ? "" : "un");
	prompt();
}

void varset_timeinc(Variable *var) {
	iprintf("Default %s set to %d.\n",
		(var->name[0] == 't') ? "time" :
		(var->name[0] == 'i') ? "increment" : "rated status",
		var->number);
	prompt();
}

void varset_style(Variable *var) {
	iprintf("Style %d set.\n", var->number);
	prompt();
}

void varset_tolerance(Variable *var) {
	char tmp[16];
	String *packet, *n;

	iprintf("Profanity filter set to %s.\n",
		(var->number == 0) ?	"strong" :
		(var->number == 1) ?	"weak" :
					"none");

	packet = StringNew("games_common_profanity", -1);
	packutfString(packet, packet);

	sprintf(tmp, "%d", 2 - var->number);
	n = StringNew(tmp, -1);
	packutfStringP(packet, n);

	nprintropString(ROP_OPTION, packet);

	StringFree(packet);
	StringFree(n);
}

void tvarset_timeinc(Variable *var) {
	Table *table = tables[primary];
	int time = 0, inc = 0;
	char limit = 0;
	char packet[9];

	if (!strcmp(var->name, "time")) {
		time = var->number * 60000;
		inc = tableinc(table) * 1000;
		if (findOption(table->options, "pl") == NULL)
			limit = 1;
	} else if (!strcmp(var->name, "increment")) {
		time = tabletime(table) * 60000;
		inc = var->number * 1000;
		if (findOption(table->options, "pl") == NULL)
			limit = 1;
	} else {
		time = tabletime(table) * 60000;
		inc = tableinc(table) * 1000;
		if (var->number)
			limit = 1;
	}

	if (time == 0) {
		time = -1;
		inc = 0;
	}

	packet[0] = limit;

	time = htonl(time);
	memcpy(&packet[1], &time, 4);

	inc = htonl(inc);
	memcpy(&packet[5], &inc, 4);

	nprinttop((uchar)primary, (char)TOP_TIME, packet, sizeof(packet));
}

void tvarset_rated(Variable *var) {
	char setting = (char)(var->number ? 1 : 0);

	nprinttop((uchar)primary, (char)TOP_RATED, &setting, 1);
}

void tvarset_protection(Variable *var) {
	char setting = (char)var->number;

	nprinttop((uchar)primary, (char)TOP_PROTECTION, &setting, 1);
}
