#!/usr/bin/env python

# This file is part of Window-Switch.
# Copyright (c) 2009-2013 Antoine Martin <antoine@nagafix.co.uk>
# Window-Switch is released under the terms of the GNU GPL v3

import gtk
import gobject

from winswitch.ui.config_common import new_config_window, add_tableline
from winswitch.ui.config_common import table_init
from winswitch.ui.session_start_options_base import SessionStartOptionsBase


class SessionOptions(SessionStartOptionsBase):
	"""
	This dialog allows the user to select the session options.
	Unlike SessionStartOptions, this dialog uses a fixed session_type.
	This is currently used for selecting shadow options.

	It only allows the user to select:
	* the screen size and bit depth (if applicable)
	* session type specific options (ie: jpeg compression for xpra, ..)
	"""

	def __init__(self, title, icon, server, client_util, command_type, shadowed_session, callback):
		SessionStartOptionsBase.__init__(self, command_type, shadowed_session)
		self.title = title
		self.icon = icon
		self.server = server
		self.client_util = client_util
		self.callback = callback
		self.show()


	def get_title(self):
		return	self.title

	def create_window(self):
		window = new_config_window(self.get_title())
		#Contents
		self.create_form_box()

		window.set_border_width(15)
		window.add(self.box)
		window.set_geometry_hints(self.box)
		window.connect('delete_event', self.close_window)
		self.ui_util.add_close_accel(window, self.close_window)
		return	window

	def create_form_box(self):
		self.box = gtk.VBox(False, 0)
		self.box.set_spacing(15)
		hbox = gtk.HBox(False, 0)
		# icon
		if self.icon:
			hbox.pack_start(gtk.image_new_from_pixbuf(self.icon))
		# title
		hbox.pack_start(self.ui_util.make_label(self.get_title(), "sans 16"))
		self.box.pack_start(hbox)

		table = table_init(self)
		self.add_start_options(table)
		self.box.pack_start(table)

		self.box.pack_start(self.create_button_box())
		self.box.show_all()

	def add_start_options(self, table):
		# virt specific options:
		def option_changed(key, value):
			self.sdebug("current options=%s" % self.session_start_options, key, value)
			self.session_start_options[key] = value
		option_widgets = self.client_util.get_options_widgets(self.server, None, self.shadowed_session, self.session_start_options, self.client_util.get_default_options(), option_changed)
		self.sdebug("option_widgets=%s" % str(option_widgets))
		for label,widget in option_widgets:
			add_tableline(self, table, label, widget)

	def create_button_box(self):
		button_box = gtk.HBox(True, 10)
		button_box.pack_start(self.ui_util.make_imagebutton("Cancel", "close", None, self.close_window))
		self.start_button = self.ui_util.make_imagebutton("Start", "play", None, self.start_session)
		button_box.pack_start(self.start_button)
		return	button_box

	def start_session(self, *args):
		self.close_window()
		self.sdebug("calling: %s(%s)" % (self.callback, self.session_start_options), *args)
		gobject.idle_add(self.callback, self.session_start_options)

	def do_populate_form(self):
		pass

	def get_selected_session_type(self):
		return	self.client_util.session_type

	def get_selected_client_util(self):
		return	self.client_util
