/*--------------------------------------------------------------------------*\

    FILE....: TONED.H
    TYPE....: C Header File
    AUTHOR..: David Rowe
    DATE....: 19/2/98

    Tone detector module.

\*--------------------------------------------------------------------------*/

#ifndef __VT_TONED_H__
#define __VT_TONED_H__

#include <stdint.h>

/*
   To program the tone detector, messages are passed to this module
   via the toned_add_tone function.  The format of the message is as
   follows:

   ushort	length
   ushort	message_type = PC_TONED_STATE_TABLE
   ushort	id		// object id

   ushort	nstates		// number of cadence states
   ushort	tone_id		// unique ID for this tone
   ushort	ntones		// number of tones (1 or 2)
   ushort	f1		// freq of first tone (Hz)
   ushort	bw1		// bandwidth of first tone (Hz)
   ushort	f2		// freq of first tone (Hz)
   ushort	bw2		// bandwidth of second tone (Hz)
   short       	A1		// min amp of 1st tone (dB) rel to 0dBm0
   short	A2		// min amp of 2nd tone (dB) rel to 0dBm0
   short	twist		// max difference in tone levels (dB)
   short	snr		// min signal to noise ratio (0-30dB)
   ushort	glitch		// deglitch period in ms

   STATE 	state_0		// first cadence state
   STATE	state_1
   .		.
   .            .
   STATE	state_states-1



   Notes:
   ------

   1. To detect a tone (pair) it must pass certain requirements (level,
      freq etc), then conform to a certain cadence pattern which is tested
      using a cadence state machine.

   2. Each channel can have up to MS (callprog.c) state machines, each
      state machine can have up to MAX_ST (callprog.c) states.

   3. The state machine starts at state 0 (the idle state), and with each
      state transition moves to the next state, ie from state n to state
      n+1.

   4. To jump to the next state, the conditions described in the state
      transition table (TONED_STATE below) for that state must be met.

   5. There are 3 types of state transitions:

      timer.: When the ON time exceeds tfire, state transition occurs. If
	      a rising or falling edge occurs, the state machine returns to
	      idle.

      rising: When a rising edge in the tone energy occurs, state transition
	      occurs if the OFF time of the tone was betweeen tmin and tmax.
	      If the length of the tone is outside these constraints, the
	      state machine returns to the idle state.  If tmin and tmax
	      equal zero, the constraints are ignored and state transition
	      occurs on the rising edge of the tone energy, regardless of
	      the off time.

      falling: As per rising, except the ON time is measured after a falling
	       edge occurs.

      delay..: The state machine remains in thsi state for the specified time


   6. The time units are in TICKS (#define in toned.c), which is set to
      1ms.  This value is also handy as conversion to ms and back is simple.
      The actual analysis is performed on 20ms frames, with a 256 point DFT,
      thus the time resolution is limited to 20ms, although defined in 1ms
      increments for convenience.
*/

/*
   State transition table consists of one entry for each state transition.
*/

#define	TIMER 	0
#define	RISING	1
#define	FALLING	2
#define	DELAYST	3

typedef struct {
    uint16_t  type;	    /* TIMER, RISING, or FALLING	    */
    uint16_t  tfire;	    /* timer mode only			    */
    uint16_t  tmin;	    /* minimum tone on/off time (non timer) */
    uint16_t  tmax;	    /* maximum tone on/off time (non timer) */
} STRAN;

struct TD;

void toned_open(struct TD **td, unsigned short ch,
                void (*callback)(uint16_t *mess,void *board), void *board);
void toned_close(struct TD *td);
void toned_add_tone(struct TD *td, uint16_t *message);
void toned_del_tone(struct TD *td, uint16_t *message);
void toned_analyse(struct TD *td, short inp[], unsigned short n);
void toned_start_logging(struct TD *td, uint16_t *m);

#endif	/* __VT_TONED_H__ */
