/*
 * Copyright (C) 2023, KylinSoft Co., Ltd.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 *
 */
#include "upm_gsettings.h"

#include <QDebug>

Q_GLOBAL_STATIC(UpmGsettings, s_upmGsettings)

UpmGsettings::UpmGsettings(QObject *parent) : QObject(parent)
{
    const QByteArray gsettingsID("org.ukui.power-manager");
    m_gsettingsConfig = new QGSettings(gsettingsID);

    m_allGsettingsKey = m_gsettingsConfig->keys();

    initGsettingsConfig();

    connect(m_gsettingsConfig, &QGSettings::changed, this, &UpmGsettings::dealGsettingsChanged);
}

UpmGsettings::~UpmGsettings()
{
    delete m_gsettingsConfig;
}

UpmGsettings *UpmGsettings::self()
{
    return s_upmGsettings;
}

void UpmGsettings::dealGsettingsChanged(const QString &key)
{
    qDebug() << "gsetting key:" << key << "changed!";
    if (true == m_qHashValue.contains(key)) {
        m_qHashValue[key] = m_gsettingsConfig->get(key);
        emit (this->*m_qHashSignalsFuns[key])(m_qHashValue[key]);
    } else {
        qDebug() << "this key:" << key << "is not monitored!";
    }
}

void UpmGsettings::setGsettingsConfig(const QString &configName, QVariant value)
{
    qDebug() << "set key:" << configName;
    if (true == m_qHashValue.contains(configName)) {
        m_qHashValue[configName] = value;
        m_gsettingsConfig->set(configName, value);
    } else {
        qDebug() << "this key:" << configName << "is not monitored!";
    }
}

bool UpmGsettings::getGsettingsConfig(const QString &configName, QVariant &value)
{
    qDebug() << "get key:" << configName;
    if (true == m_qHashValue.contains(configName)) {
        value = m_qHashValue[configName];
        return true;
    }
    qDebug() << "this key:" << configName << "is not monitored!";
    return false;
}

void UpmGsettings::addConfigMonitor(const QString &configName, pSignalFun signalFun, QVariant::Type type)
{
    qDebug() << "add config" << configName;
    if (m_allGsettingsKey.contains(configName)) {
        m_qHashValue[configName] = m_gsettingsConfig->get(configName);
        m_qHashSignalsFuns[configName] = signalFun;
        m_qHashValueType[configName] = type;
        qDebug() << configName << "to be monitored!";
    }
}

QVariant::Type UpmGsettings::getValueType(const QString &configName)
{
    if (m_qHashValueType.contains(configName)) {
        return m_qHashValueType[configName];
    }
    return QVariant::Invalid;
}

void UpmGsettings::initGsettingsConfig()
{
    addConfigMonitor(GSETTINGS_KEY_POWER_POLICY_AC,
                     &UpmGsettings::powerPolicyAcChanged,
                     QVariant::Int);
    addConfigMonitor(GSETTINGS_KEY_POWER_POLICY_BATTERY,
                     &UpmGsettings::powerPolicyBatteryChanged,
                     QVariant::Int);
    addConfigMonitor(GSETTINGS_KEY_POWER_POLICY_BATTERY_BACKUP,
                     &UpmGsettings::powerPolicyBatteryBackupChanged,
                     QVariant::Int);

    addConfigMonitor(GSETTINGS_KEY_SLEEP_DISPLAY_AC,
                     &UpmGsettings::sleepDisplayAcChanged,
                     QVariant::Int);
    addConfigMonitor(GSETTINGS_KEY_SLEEP_DISPLAY_BATTERY,
                     &UpmGsettings::sleepDisplayBatteryChanged,
                     QVariant::Int);

    addConfigMonitor(GSETTINGS_KEY_SLEEP_COMPUTER_AC,
                     &UpmGsettings::sleepComputerAcChanged,
                     QVariant::Int);
    addConfigMonitor(GSETTINGS_KEY_SLEEP_COMPUTER_BATTERY,
                     &UpmGsettings::sleepComputerBatteryChanged,
                     QVariant::Int);

    addConfigMonitor(GSETTINGS_KEY_BUTTON_LID_AC,
                     &UpmGsettings::buttonLidAcChanged,
                     QVariant::String);
    addConfigMonitor(GSETTINGS_KEY_BUTTON_LID_BATTERY,
                     &UpmGsettings::buttonLidBatteryChanged,
                     QVariant::String);

    addConfigMonitor(GSETTINGS_KEY_PERCENTAGE_LOW,
                     &UpmGsettings::percentageLowChanged,
                     QVariant::Int);
    //极低电量百分比
    addConfigMonitor(GSETTINGS_KEY_PERCENTAGE_ACTION,
                     &UpmGsettings::percentageActionChanged,
                     QVariant::Int);
    //极低电量操作
    addConfigMonitor(GSETTINGS_KEY_ACTION_CRITICAL_BATTERY,
                     &UpmGsettings::actionCriticalBatteryChanged,
                     QVariant::String);

    addConfigMonitor(GSETTINGS_KEY_BATTERY_SAVE_SWITCH,
                     &UpmGsettings::batterySaveSwitchChanged,
                     QVariant::Bool);
    addConfigMonitor(GSETTINGS_KEY_LOW_BATTERY_AUTO_SAVE,
                     &UpmGsettings::lowBatteryAutoSaveChanged,
                     QVariant::Bool);
    addConfigMonitor(GSETTINGS_KEY_BATTERY_SAVE_REDUCE_BRIGHTNESS,
                     &UpmGsettings::batterySaveReduceBrightnessChanged,
                     QVariant::Bool);

    addConfigMonitor(GSETTINGS_KEY_BRIGHTNESS_AC,
                     &UpmGsettings::brightnessAcChanged,
                     QVariant::Int);
}
