/*===========================================================================
*
*                            PUBLIC DOMAIN NOTICE
*               National Center for Biotechnology Information
*
*  This software/database is a "United States Government Work" under the
*  terms of the United States Copyright Act.  It was written as part of
*  the author's official duties as a United States Government employee and
*  thus cannot be copyrighted.  This software/database is freely available
*  to the public for use. The National Library of Medicine and the U.S.
*  Government have not placed any restriction on its use or reproduction.
*
*  Although all reasonable efforts have been taken to ensure the accuracy
*  and reliability of the software and data, the NLM and the U.S.
*  Government do not and cannot warrant the performance or results that
*  may be obtained by using this software or data. The NLM and the U.S.
*  Government disclaim all warranties, express or implied, including
*  warranties of performance, merchantability or fitness for any particular
*  purpose.
*
*  Please cite the author in any work or product based on this material.
*
* ===========================================================================
*
*/

#ifndef _h_vdb_vdb_priv_
#define _h_vdb_vdb_priv_

#ifndef _h_vdb_extern_
#include <vdb/extern.h>
#endif

#ifndef _h_klib_defs_
#include <klib/defs.h>
#endif

#include <stdarg.h>

#ifdef __cplusplus
extern "C" {
#endif


/*--------------------------------------------------------------------------
 * forwards
 */
struct KDatabase;
struct KTable;
struct KMetadata;
struct KDBManager;
struct VDBManager;
struct VSchema;
struct VTypedef;
struct VDatabase;
struct VTable;
struct VCursor;
struct VCursorParams;
struct VXformInfo;
struct KDataBuffer;
struct KNamelist;


/*--------------------------------------------------------------------------
 * VDBManager
 */

/* OpenKDBManager
 *  returns a new reference to KDBManager used by VDBManager
 */
VDB_EXTERN rc_t CC VDBManagerOpenKDBManagerRead ( struct VDBManager const *self,
    struct KDBManager const **kmgr );
VDB_EXTERN rc_t CC VDBManagerOpenKDBManagerUpdate ( struct VDBManager *self,
    struct KDBManager **kmgr );

#define VDBManagerGetKDBManagerRead VDBManagerOpenKDBManagerRead
#define VDBManagerGetKDBManagerUpdate VDBManagerOpenKDBManagerUpdate


/*--------------------------------------------------------------------------
 * VSchema
 */

/* DescribeTypedef - PRIVATE
 *  a type is defined as either:
 */
VDB_EXTERN rc_t CC VSchemaDescribeTypedef ( struct VSchema const *self,
    struct VTypedef *def, uint32_t type_id );

/* LastIntrinsicTypeId - PRIVATE
 *  used to allow cast to intrinsics
 */
VDB_EXTERN uint32_t CC VSchemaLastIntrinsicTypeId ( struct VSchema const *self );

/* ListLegacyTables - PRIVATE
 *  scan all tables defined in schema
 *  return the fully-qualified and versioned name of each legacy SRA table
 */
VDB_EXTERN rc_t CC VSchemaListLegacyTables ( struct VSchema const *self, struct KNamelist **list );


/*--------------------------------------------------------------------------
 * VDatabase
 */

/* OpenKDatabase
 *  returns a new reference to underlying KDatabase
 */
VDB_EXTERN rc_t CC VDatabaseOpenKDatabaseRead ( struct VDatabase const *self, struct KDatabase const **kdb );
VDB_EXTERN rc_t CC VDatabaseOpenKDatabaseUpdate ( struct VDatabase *self, struct KDatabase **kdb );



/*--------------------------------------------------------------------------
 * VTable
 */

/* OpenKTable
 *  returns a new reference to underlying KTable
 */
VDB_EXTERN rc_t CC VTableOpenKTableRead ( struct VTable const *self, struct KTable const **ktbl );
VDB_EXTERN rc_t CC VTableOpenKTableUpdate ( struct VTable *self, struct KTable **ktbl );

#define VTableGetKTableRead VTableOpenKTableRead
#define VTableGetKTableUpdate VTableOpenKTableUpdate
    
VDB_EXTERN rc_t CC VTableDropColumn(struct VTable *self, const char *name, ...);
VDB_EXTERN rc_t CC VTableVDropColumn(struct VTable *self, const char *name, va_list args);
    

/* VUntypedTableTest
 *  support for tables created before embedded schema
 *
 *  "ktbl" [ IN ] - physical table to recognize
 *
 *  "meta" [ IN ] - open metadata for table
 *
 *  returns true if table is recognized
 */
typedef bool ( CC * VUntypedTableTest )
   ( struct KTable const *ktbl, struct KMetadata const *meta );


/* StoreSchema
 *  stores current schema definition in metadata
 *
 *  useful if table was opened using a later version of schema than
 *  was used for its creation.
 */
rc_t VTableStoreSchema ( struct VTable *self );


/*--------------------------------------------------------------------------
 * VCursor
 */

/* PermitPostOpenAdd
 *  allows columns to be added to open cursor
 *  for write cursor, the effect lasts until the first row commit
 */
VDB_EXTERN rc_t CC VCursorPermitPostOpenAdd ( struct VCursor const *self );

/*  VCursorGetSchema
 *  returns current schema of the open cursor
 */
VDB_EXTERN struct VSchema const* CC VCursorGetSchema ( struct VCursor const *self);

/* PageIdRange
 *  returns id range for page
 *
 *  "id" [ IN ] - page containing this row id is target
 *
 *  "first" [ OUT, NULL OKAY ] and "last" [ OUT, NULL OKAY ] -
 *  id range is returned in these output parameters, where
 *  at least ONE must be NOT-NULL
 */
VDB_EXTERN rc_t CC VCursorPageIdRange ( struct VCursor const *self,
    uint32_t idx, int64_t id, int64_t *first, int64_t *last );




/*--------------------------------------------------------------------------
 * VCursorParams
 */

/* Get
 *  return KDataBuffer value by reference
 */
VDB_EXTERN rc_t CC VCursorParamsGet ( struct VCursorParams const *self,
    const char *name, struct KDataBuffer **value );

/* Set
 *  set a textual parameter value
 */
VDB_EXTERN rc_t CC VCursorParamsSet( struct VCursorParams const *self,
    const char *name, const char *fmt, ... );
VDB_EXTERN rc_t CC VCursorParamsVSet ( struct VCursorParams const *self,
    const char *name, const char *fmt, va_list args );

/* Unset
 *  set the value of a named parameter to "undefined"
 */
VDB_EXTERN rc_t CC VCursorParamsUnset ( struct VCursorParams const *self, const char *name );


/*--------------------------------------------------------------------------
 * xform
 */

/*
  uses vftReserved code
  returns little-endian data, no row-wise rle
  fixed row-length is calculated as dst->elem_count / num_rows
 */
#define vftLegacyBlob vftReserved

typedef struct VLegacyBlobResult VLegacyBlobResult;
struct VLegacyBlobResult
{
    struct KDataBuffer *dst;
    uint8_t byte_order;
};

typedef rc_t ( CC * VNoHdrBlobFunc ) ( void *self,
    struct VXformInfo const *info, VLegacyBlobResult *rslt,
    struct KDataBuffer const *src );

/* function to call to test table against schema */
typedef bool ( CC * VUntypedFunc )
    ( struct KTable const *tbl, struct KMetadata const *meta );



#ifdef __cplusplus
}
#endif

#endif /* _h_vdb_vdb_priv_ */
