/*  Copyright (c) 2005 Romain BONDUE
    This file is part of RutilT.

    RutilT is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    RutilT is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with RutilT; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
/** \file GtkGUI.cxx
    \author Romain BONDUE
    \date 23/07/2005 */
#include <sstream>
#include <algorithm> // find()
#include <vector>
#include <limits>
#include <csignal>
#include <cstring> // std::memset()
#ifndef NDEBUG
#include <iostream>
#endif // NDEBUG

extern "C"{
#include <unistd.h> // ::getuid()
#include <gdk-pixbuf/gdk-pixbuf.h>
}

#include "GtkGUI.h"
#include "IDriver.h"
#include "Module.h"
#include "Parameters.h"
#include "Su.h"
#include "UserData.h"
#include "NetTools.h"
#include "IfaceRoot.h"
#include "WWidgets.h"
#include "OptionsFile.h"
#include "ErrorsCode.h"
#include "StaticSettings.h"



namespace
{
    void TriggerResponseCallBack (::GtkEntry*, ::GtkDialog* pDialog) throw()
    {
        ::gtk_dialog_response (pDialog, GTK_RESPONSE_OK);

    } // TriggerResponseCallBack()


    void SetFocusCallBack (::GtkEntry*, ::GtkWidget* pWidget) throw()
    {
        ::gtk_widget_grab_focus (pWidget);

    } // SetFocusCallBack()


    const unsigned NbColumn (3);
    const char* ColumnTitle [NbColumn] = {"Interface", "Device name",
                                          "Special support"};

    class CInterfaceSelector
    {
      private :
        ::GtkWidget* m_pListView;
        unsigned m_SelectedRow;

        static bool RowSelectedCallBack (::GtkTreeSelection*, ::GtkTreeModel*,
                                         ::GtkTreePath* pTreePath, bool,
                                         CInterfaceSelector* This) throw()
        {
            This->m_SelectedRow = ::gtk_tree_path_get_indices (pTreePath) [0];
            return true;

        } // RowSelectedCallBack()


      public :
        CInterfaceSelector
            (const std::vector<nsWireless::CDeviceInfo>& DevicesVec) throw()
            : m_pListView (0), m_SelectedRow (0)
        {
                // Create UI componenents :
            ::GtkListStore* const pListStore
                           (GTK_LIST_STORE (::gtk_list_store_new (NbColumn,
                                G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING)));
            m_pListView = ::gtk_tree_view_new_with_model (GTK_TREE_MODEL
                                                                (pListStore));
            GtkCellRenderer* const pRenderer (::gtk_cell_renderer_text_new());
            for (unsigned i (0) ; i < NbColumn ; ++i)
                ::gtk_tree_view_append_column (GTK_TREE_VIEW (m_pListView),
                    ::gtk_tree_view_column_new_with_attributes
                                        (ColumnTitle [i], pRenderer, "text", i,
                                         static_cast<void*> (0)));
                // Callbacks :
            ::gtk_tree_selection_set_select_function
                (::gtk_tree_view_get_selection (GTK_TREE_VIEW (m_pListView)),
                 reinterpret_cast< ::GtkTreeSelectionFunc>
                                            (RowSelectedCallBack), this, 0);
                // Put data :
            for (unsigned i (0) ; i < DevicesVec.size() ; ++i)
            {
                    // Create the row :
                ::GtkTreeIter Iter;
                ::gtk_list_store_insert (pListStore, &Iter, i);
                    // Set the data :
                ::gtk_list_store_set (pListStore, &Iter,
                                  0, DevicesVec [i].GetDeviceName().c_str(),
                                  1, DevicesVec [i].GetProtoName().c_str(),
                                  2, nsCore::IsSupported (DevicesVec [i])
                                                        ? "yes" : "no", -1);
            }
                // Cleaning :
            ::g_object_unref (pListStore);

        } // CInterfaceSelector()


        operator ::GtkWidget* () throw()
        {
            return m_pListView;
        
        } // operator ::GtkWidget*()


        unsigned GetSelectedRow () const throw()
        {
            return m_SelectedRow;
        
        } // GetSelectedRow()

    }; // CInterfaceSelector


    int SelectInterface
                    (::GtkWindow* pMainWindow,
                     const std::vector<nsWireless::CDeviceInfo>& DevicesVec)
                                                throw (nsErrors::CException)
    {
        ::GtkWidget* const pDialog (::gtk_dialog_new_with_buttons (
                                                    "Interface selection",
                                                    0, ::GtkDialogFlags (0),
                                                    GTK_STOCK_CANCEL,
                                                    GTK_RESPONSE_DELETE_EVENT,
                                                    GTK_STOCK_OK,
                                                    GTK_RESPONSE_OK,
                                                    static_cast<void*> (0)));
        ::gtk_window_set_resizable (GTK_WINDOW (pDialog), false);
        ::gtk_dialog_set_has_separator (GTK_DIALOG (pDialog), false);
        if (pMainWindow)
            ::gtk_window_set_transient_for (GTK_WINDOW (pDialog),
                                            GTK_WINDOW (pMainWindow));
        ::gtk_container_set_border_width (GTK_CONTAINER (pDialog), 6);
        ::gtk_container_add (GTK_CONTAINER (GTK_DIALOG (pDialog)->vbox),
                             ::gtk_label_new ("Select the interface to use"
                                              " :"));
        CInterfaceSelector IfaceSelector (DevicesVec);
        ::gtk_box_pack_start_defaults (GTK_BOX (GTK_DIALOG (pDialog)->vbox),
                                       IfaceSelector);
        ::gtk_widget_show_all (GTK_DIALOG (pDialog)->vbox);

        if (::gtk_dialog_run (GTK_DIALOG (pDialog)) !=
                                                    GTK_RESPONSE_DELETE_EVENT)
        {
            ::gtk_widget_destroy (pDialog);
            return IfaceSelector.GetSelectedRow();
        }
        else
        {
            ::gtk_widget_destroy (pDialog);
            return -1;
        }

    } // SelectInterface()


    inline bool IsReallyRoot () throw()
    {
        return !::getuid();
    
    } // IsReallyRoot()


    void SignalsCallback (int) throw()
    {
        if (::gtk_main_level())
            ::gtk_main_quit();

    } // SignalsCallback()


    void TrapSignals () throw()
    {
        struct sigaction Settings;
        std::memset (&Settings, 0, sizeof (struct sigaction));
        Settings.sa_handler = SignalsCallback;
            // If those calls fail, we can live with it.
        ::sigaction (SIGINT, &Settings, 0);
        ::sigaction (SIGTERM, &Settings, 0);

    } // TrapSignals()



    const unsigned SiteSurveyTimerPeriod (1000);
    const unsigned GlobalTimerPeriod (3000);
    const unsigned InvalidProfileI (std::numeric_limits<unsigned>::max());

    const char* const IfaceStatusLabelDown ("down");
    const char* const IfaceStatusLabelUp ("up");

        // FIXME Horrible... Only used by the tray icon.
        // That could make sense however, the main window is a singleton.
    nsGUI::CGtkGUI* GlobalUglyThis;

} // anonymous namespace



void nsGUI::NotifyError (const std::string& Msg, int Code,
                         ::GtkWindow* pWindow) throw()
{
    ::GtkWidget* const pDialog (::gtk_message_dialog_new (pWindow,
                        GTK_DIALOG_MODAL, GTK_MESSAGE_ERROR,
                        GTK_BUTTONS_OK, Code ? "%s\nCode : %d" : Msg.c_str(),
                        Msg.c_str(), Code));
    ::gtk_dialog_run (GTK_DIALOG (pDialog));
    ::gtk_widget_destroy (pDialog);

} // NotifyError()


bool nsGUI::AskUser (const std::string& Msg, ::GtkWindow* pWindow) throw()
{
    ::GtkWidget* const pDialog (::gtk_message_dialog_new (pWindow,
                        GTK_DIALOG_MODAL, GTK_MESSAGE_QUESTION,
                        GTK_BUTTONS_YES_NO, Msg.c_str()));
    const int Result (::gtk_dialog_run (GTK_DIALOG (pDialog)));
    ::gtk_widget_destroy (pDialog);
    return Result == GTK_RESPONSE_YES;

} // AskUser()


nsGUI::CGtkGUI::CGtkGUI () throw (nsErrors::CException)
    : m_pModule (0), m_pWindow (GTK_WINDOW (::gtk_window_new
                                                        (GTK_WINDOW_TOPLEVEL))),
      m_pNoteBook (GTK_NOTEBOOK (::gtk_notebook_new())), m_Exc ("", 0),
      m_GlobalTimer (0), m_SiteSurveyTab (this), m_SiteSurveyTabTimer (0),
      m_TimeRemaining (0), m_ProfileTab (this),
      m_TrayIcon (nsCore::TrayIconDisconnectedPath,
                  G_CALLBACK (TrayIconClickedCallBack), this),
      m_Flags (IsReallyRoot() ? IsRootF : 0U),
#ifdef NHELPER
      m_PrefsPage (true),
#else
      m_PrefsPage (IsRoot()),
#endif // NHELPER
      m_LeaveRootModeTimer (0), m_CurrentRateIndex (0), m_pTrayMenu (0)
{
    ::gtk_window_set_position (m_pWindow, GTK_WIN_POS_CENTER);
    ::gtk_window_set_resizable (m_pWindow, false);
    ::gtk_notebook_append_page (m_pNoteBook, m_LinkStatusTab,
                                ::gtk_label_new ("Link status"));
    ::gtk_notebook_append_page (m_pNoteBook, m_StatsPage,
                                ::gtk_label_new ("Statistics"));
    ::gtk_notebook_append_page (m_pNoteBook, m_ProfileTab,
                                ::gtk_label_new ("Profiles"));
    ::gtk_notebook_append_page (m_pNoteBook, m_SiteSurveyTab,
                                ::gtk_label_new ("Site survey"));
    ::gtk_notebook_append_page (m_pNoteBook, m_PrefsPage,
                                ::gtk_label_new ("Options"));
    ::gtk_container_add (GTK_CONTAINER (m_pWindow), GTK_WIDGET (m_pNoteBook));

    if (IsRoot())
        m_PrefsPage.SetRootMode (true);
    m_PrefsPage.AddChangeInterfaceCallBack (G_CALLBACK
                                                    (ChangeInterfaceCallBack),
                                            this);
    m_PrefsPage.AddAutoLeaveRootModeCallBack (G_CALLBACK
                                                (AutoLeaveRootModeCallBack),
                                              this);
    m_PrefsPage.AddRootModeChangedCallBack (G_CALLBACK
                                                    (RootModeChangedCallBack),
                                            this);
    m_PrefsPage.AddRateChangedCallBack (G_CALLBACK (RateChangedCallBack),
                                        this);
    m_PrefsPage.AddTrayIconButtonCallBack
                            (G_CALLBACK (DisplayTrayIconButtonCallBack), this);
    m_PrefsPage.AddStartMinimizedCallBack
                            (G_CALLBACK (StartMinimizedButtonCallBack), this);
    m_PrefsPage.AddIfaceStatusChangeCallBack
                                (G_CALLBACK (IfaceStatusChangedCallBack), this);

    TrapSignals();
    LoadOptions();
    SetWindowIcons();
#ifndef NDEBUG
    std::cerr << "GUI successfully initialized.\nIs launched as root : "
              << std::boolalpha << IsRoot() << std::noboolalpha << std::endl;
#endif // NDEBUG
    GlobalUglyThis = this;

} // CGtkGUI()


nsGUI::CGtkGUI::~CGtkGUI () throw()
{
    ::gtk_widget_destroy (GTK_WIDGET (m_pWindow));
    if (m_GlobalTimer)
        ::g_source_remove (m_GlobalTimer);
    if (m_SiteSurveyTabTimer)
        ::g_source_remove (m_SiteSurveyTabTimer);
    if (m_pTrayMenu)
        ::gtk_object_destroy (GTK_OBJECT (m_pTrayMenu));

} // ~CGtkGUI()


void nsGUI::CGtkGUI::SetWindowIcons () throw()
{
    ::GList* pIconsList (0);
    ::GError* pError (0);
    pIconsList = ::g_list_append (pIconsList, ::gdk_pixbuf_new_from_file
                                        (nsCore::Icon16Path.c_str(), &pError));
    pIconsList = ::g_list_append (pIconsList, ::gdk_pixbuf_new_from_file
                                        (nsCore::Icon32Path.c_str(), &pError));
    pIconsList = ::g_list_append (pIconsList, ::gdk_pixbuf_new_from_file
                                        (nsCore::Icon48Path.c_str(), &pError));
    pIconsList = ::g_list_append (pIconsList, ::gdk_pixbuf_new_from_file
                                        (nsCore::Icon64Path.c_str(), &pError));
    pIconsList = ::g_list_append (pIconsList, ::gdk_pixbuf_new_from_file
                                    (nsCore::Icon128Path.c_str(), &pError));
    if (pError)
    {
        NotifyError (std::string ("Could not load application icons : ") +
                                    pError->message, pError->code, m_pWindow);
        ::g_error_free (pError);
    }
    else
        ::gtk_window_set_default_icon_list (pIconsList);
    ::g_list_free (pIconsList);

} // SetWindowIcons()


void nsGUI::CGtkGUI::Run (const nsCore::CCLIOpt& Opt)
                                                    throw (nsErrors::CException)
{
#ifndef NDEBUG
    std::cerr << "Loading module...\n";
#endif // NDEBUG
    try{if (!ChangeModule (Opt.IfaceName())) return;}
    catch (const nsErrors::CException& Exc)
    {
#ifndef NDEBUG
        std::cerr << "Couldn't load module : " << Exc.GetMsg() << ' '
                  << Exc.GetCode() << std::endl;
#endif // NDEBUG
        if (Exc.GetCode() == m_Exc.GetCode() || ProcessError (Exc) ||
                                                            !m_pModule.get())
            throw;
    }
        // We can't set this signal and "delete_event" before having a module.
    ::g_signal_connect (G_OBJECT (m_pNoteBook), "switch-page",
                        G_CALLBACK (PageSwitchCallBack), this);

    if (!m_PrefsPage.StartMinimized() && !Opt.Hidden())
        ::gtk_widget_show_all (GTK_WIDGET (m_pWindow));
    if (m_PrefsPage.DisplayTrayIcon())
    {
        m_TrayIcon.Show();
        m_ExitHandlerID = ::g_signal_connect_swapped (G_OBJECT (m_pWindow),
                            "delete_event", G_CALLBACK (HideCallBack), this);
    }
    else
        m_ExitHandlerID = ::g_signal_connect (G_OBJECT (m_pWindow),
                                "delete_event", G_CALLBACK (DeleteCallBack), 0);

    if (Opt.Profile())
    {
        unsigned I (0);
        for ( ; I < m_pModule->GetNbProfile() ; ++I)
            if (m_pModule->GetProfile (I).GetName() == Opt.ProfileName())
            {
                if (BecomeRoot())
                {
#ifndef NDEBUG
                    std::cerr << "Applying the profile (command line)...\n";
#endif // NDEBUG
                    m_pModule->ApplyProfile (I);
                }
#ifndef NDEBUG
                else
                    std::cerr << "Cannot apply the profile (command line) :"
                                 " not root.\n";
#endif // NDEBUG
                break;
            }
        if (I == m_pModule->GetNbProfile())
            throw (nsErrors::CException (std::string ("Profile not found : ") +
                                 Opt.ProfileName(), nsErrors::ProfileNotFound));
    }
    if (Opt.Dhcp() && BecomeRoot())
    {
#ifndef NDEBUG
        std::cerr << "Running DHCP client (command line option).\n";
#endif // NDEBUG
        m_pIface->SetIPFromDHCP (m_pModule->GetDriver().GetSSID());
    }

#ifndef NDEBUG
    std::cerr << "Running main loop...\n";
#endif // NDEBUG
    ::gtk_main();
#ifndef NDEBUG
    std::cerr << "Exiting main loop...\n";
#endif // NDEBUG
    if (m_Exc.GetCode())
    {
#ifndef NDEBUG
        std::cerr << "Error.\n";
#endif // NDEBUG
        throw nsErrors::CException (m_Exc);
    }

} // Run()


void nsGUI::CGtkGUI::PageSwitchCallBack (::GtkNotebook* Notebook,
                                         ::GtkNotebookPage* pPage,
                                         PageNum_e PageNum, CGtkGUI* This)
                                                                        throw()
{       /* WARNING TrayIconClickedCallBack() calls this function with a null
           pointer for the GtkNotebookPage. */
    try
    {
        switch (PageNum)
        {
          case LinkStatus :
            This->RefreshLinkStatus();
          break;

          case Stats :
            This->IsStatsAvailable (This->RefreshStats());
          break;

          case SiteSurvey :
            This->SelectCurrentCell();
          //break;

          default :; // To avoid a warning.
        }
    }
    catch (const nsErrors::CException& Exc) {This->ProcessError (Exc);}

} // PageSwitchCallBack()


bool nsGUI::CGtkGUI::GlobalTimerWithTrayCallBack (CGtkGUI* This) throw()
{
    try
    {
        nsWireless::CMacAddress APMacAddr;
        try
        {
            APMacAddr = This->m_pModule->GetDriver().GetAPMacAddr();
        }
        catch (const nsErrors::CSystemExc& Exc)
        {
            if (Exc.GetCode() != ENETDOWN)
                throw;
            else
                This->IfaceIsDown();
        }
        if (APMacAddr.Empty())
        {
            This->m_TrayIcon.SetIcon (nsCore::TrayIconDisconnectedPath);
            This->m_TrayIcon.SetTooltip ("Disconnected");
            if (GTK_WIDGET_VISIBLE (GTK_WIDGET (This->m_pWindow)))
                This->m_LinkStatusTab.Clear();
            This->m_ToolTipHeader.erase();
        }
        else
        {
            if (This->m_ToolTipHeader.empty())
                This->m_ToolTipHeader =
                        This->m_pModule->GetDriver().GetProtoName() + " (" +
                    This->m_pModule->GetDriver().GetDeviceName() + ")\n\n" +
                        "ESSID : " + This->m_pModule->GetDriver().GetSSID();
            nsWireless::CQuality Quality;
            unsigned TxRate (0);
            try
            {
                Quality = This->m_pModule->GetDriver().GetQuality();
            }
            catch (...) {}
            try
            {
                TxRate = This->m_pModule->GetDriver().GetTxRate();
            }
            catch (...) {}
            This->RefreshTray (Quality, TxRate);
            if (GTK_WIDGET_VISIBLE (GTK_WIDGET (This->m_pWindow)))
                switch (::gtk_notebook_get_current_page (This->m_pNoteBook))
                {
                  case LinkStatus :
                    This->RefreshLinkStatus (Quality, TxRate, APMacAddr);
                  break;

                  case Stats :
                    if (This->IsStatsAvailable())
                        This->IsStatsAvailable (This->RefreshStats());
                  break;

                  case SiteSurvey :
                    This->SelectCurrentCell (APMacAddr);
                  //break;

                  default :; // To avoid a warning.
                }
        }
    }
    catch (const nsErrors::CException& Exc)
    {
        return !This->ProcessError (Exc);
    }
    return true; // Keep the timer running.

} // GlobalTimerWithTrayCallBack()


bool nsGUI::CGtkGUI::GlobalTimerWithoutTrayCallBack (CGtkGUI* This) throw()
{
    try
    {
        if (GTK_WIDGET_VISIBLE (GTK_WIDGET (This->m_pWindow)))
            switch (::gtk_notebook_get_current_page (This->m_pNoteBook))
            {
              case LinkStatus :
                This->RefreshLinkStatus();
              break;

              case Stats :
                if (This->IsStatsAvailable())
                    This->IsStatsAvailable (This->RefreshStats());
              break;

              case SiteSurvey :
                This->SelectCurrentCell();
              //break;

              default :; // To avoid a warning.
            }
    }
    catch (const nsErrors::CException& Exc)
    {
        return !This->ProcessError (Exc);
    }
    return true; // Keep the timer running.

} // GlobalTimerCallWithoutTrayCallBack


int nsGUI::CGtkGUI::DeleteCallBack (::GtkWidget*, ::GdkEvent*, void*) throw()
{
        // Used with a different signature by the callback of the tray menu.
    if (::gtk_main_level())
        ::gtk_main_quit();
    return true; // Don't destroy the window (the destructor will do).

} // DeleteCallBack()


bool nsGUI::CGtkGUI::Connect (unsigned CellI) throw()
{
#ifndef NDEBUG
    std::cerr << "Connecting..." << std::endl;
#endif // NDEBUG
    try
    {
        const unsigned ProfileI (CreateProfile (CellI));
        if (ProfileI != InvalidProfileI && BecomeRoot())
        {
            const nsWireless::CCell& Cell (m_CellVec [CellI]);
            try
            {
                m_pModule->ApplyProfile (ProfileI, Cell.GetAPMacAddress());
            }
            catch (const nsErrors::CException& Exc)
            {
                if (Exc.GetCode() == ENETDOWN && BringIfaceUp())
                    m_pModule->ApplyProfile (ProfileI, Cell.GetAPMacAddress());
                else
                    throw;
            }
            SetIPFromProfile (ProfileI);
            SelectMatchingProfile (Cell);
            RefreshRates();
            m_ToolTipHeader.clear();
            return true; // Select the row.
        }
    }
    catch (const std::bad_alloc& Exc)
    {
        ProcessError (nsErrors::CException (Exc.what(),
                                            nsErrors::OutOfMemory));
    }
    catch (const nsErrors::CException& Exc)
    {       // We may have been disconnected from the previous ap.
        if (!ProcessError (Exc))
            try{SelectCurrentCell();}
            catch (const nsErrors::CSystemExc&) {} // Ignored.
    }
    return false; // Don't select the row.

} // Connect()


unsigned nsGUI::CGtkGUI::CreateProfile (unsigned CellI) throw()
{
    const unsigned NProfile (m_pModule->GetNbProfile());
    ::gint ActiveProfile (0);
    if (NProfile)
    {
        ::GtkDialog* const pDialog (GTK_DIALOG (::gtk_dialog_new_with_buttons
                                                ("Connection", m_pWindow,
                                                 ::GtkDialogFlags
                                                            (GTK_DIALOG_MODAL),
                                         GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
                                         GTK_STOCK_OK, GTK_RESPONSE_OK,
                                         static_cast<void*> (0))));
        ::gtk_window_set_resizable (GTK_WINDOW (pDialog), false);
        ::gtk_box_pack_start_defaults (GTK_BOX (pDialog->vbox),
                            ::gtk_label_new ("Select which profile to use : "));
        ::GtkComboBox* const pComboBox (GTK_COMBO_BOX
                                                (::gtk_combo_box_new_text()));
        ::gtk_combo_box_append_text (pComboBox, "<Create a new profile>");
        for (unsigned I (0) ; I  < NProfile ; ++I)
        {
            const nsUserData::CProfile& Profile (m_pModule->GetProfile (I));
            ::gtk_combo_box_append_text (pComboBox, Profile.GetName().c_str());
            if (Profile.Match (m_CellVec [CellI]))
                ActiveProfile = I + 1;
        }
        ::gtk_combo_box_set_active (pComboBox, ActiveProfile);
        ::gtk_box_pack_start (GTK_BOX (pDialog->vbox), GTK_WIDGET (pComboBox),
                                                            FALSE, FALSE, 8);
        ::gtk_container_set_border_width (GTK_CONTAINER (pDialog->vbox), 100);

        ::gtk_widget_show_all (GTK_WIDGET (pDialog));
        for ( ; ; )
        {
            if (::gtk_dialog_run (pDialog) == GTK_RESPONSE_OK)
            {
                ActiveProfile = ::gtk_combo_box_get_active (pComboBox);
                if (!ActiveProfile)
                {
                    if (!NewProfile (CellI))
                        continue;
                    ActiveProfile = NProfile;
                }
                else
                    --ActiveProfile; // First raw is : <Create a new profile>
                break;
            }
            ::gtk_widget_destroy (GTK_WIDGET (pDialog));
            return InvalidProfileI;
        }
        ::gtk_widget_destroy (GTK_WIDGET (pDialog));
    }
    else if (!NewProfile (CellI)) // Will create a new profile, so return
        return InvalidProfileI;   // 0 if successful : it's now a valid index.
    return ActiveProfile;

} // CreateProfile()


void nsGUI::CGtkGUI::Scan () throw()
{
    try
    {
        if (BecomeRoot())
        {
            m_CellVec.clear();
            try
            {
                m_pModule->GetDriver().Scan();
            }
            catch (const nsErrors::CException& Exc)
            {
                if (Exc.GetCode() == ENETDOWN && BringIfaceUp())
                    m_pModule->GetDriver().Scan();
                else
                    throw;
            }
            m_SiteSurveyTab.ScanningStarted();
            m_TimeRemaining = 15000; // The driver has 15 seconds.
            m_SiteSurveyTabTimer = ::g_timeout_add (SiteSurveyTimerPeriod,
                             reinterpret_cast< ::gboolean (*) (void*)>
                                            (RefreshCellListCallBack), this);
        }
    }
    catch (const nsErrors::CException& Exc)
    {
        ProcessError (Exc);
    }

} // Scan()


bool nsGUI::CGtkGUI::RefreshCellListCallBack (CGtkGUI* This) throw()
{
#ifndef NDEBUG
    std::cerr << "CGtkGUI::RefreshCellListCallBack()\n";
#endif // NDEBUG
    This->m_TimeRemaining -= SiteSurveyTimerPeriod;
    try
    {
        if (!This->m_TimeRemaining)
            throw nsErrors::CException ("Failed to get scan result in time.",
                                        nsErrors::ScanningTimeOut);
        This->m_pModule->GetDriver().GetScanResult (This->m_CellVec);
#ifndef NDEBUG
        std::cerr << "\tGot the results.\n";
#endif // NDEBUG
        const nsWireless::CMacAddress Addr
                                (This->m_pModule->GetDriver().GetAPMacAddr());
        const nsWireless::CCell* const pCurrentCell
                    (This->m_SiteSurveyTab.AddCells (This->m_CellVec, Addr));
        if (pCurrentCell)
        {
#ifndef NDEBUG
            std::cerr << "\tSelecting matching profile.\n";
#endif // NDEBUG
            This->SelectMatchingProfile (*pCurrentCell);
        }
    }
    catch (const std::bad_alloc& Exc)
    {
        This->ProcessError (nsErrors::CException (Exc.what(),
                            nsErrors::OutOfMemory));
        return false;
    }
    catch (const nsErrors::CException& Exc)
    {
#ifndef NDEBUG
        std::cerr << "\tError : " << Exc.GetMsg() << ' ' << Exc.GetCode()
                  << std::endl;
#endif // NDEBUG
        if (Exc.GetCode() == EINTR || Exc.GetCode() == EAGAIN)
            return true; // We go for another SiteSurveyTimerPeriod.
        This->ProcessError (Exc);
    }
    This->m_SiteSurveyTab.ScanningStopped();
    This->m_SiteSurveyTabTimer = 0;
#ifndef NDEBUG
    std::cerr << "Exiting CGtkGUI::RefreshCellListCallBack()\n";
#endif // NDEBUG
    return false;

} // RefreshCellListCallBack()


bool nsGUI::CGtkGUI::ProcessError (const nsErrors::CException& Exc) throw()
{
    switch (Exc.GetCode())
    {
      case nsErrors::ScanningTimeOut :
      case nsErrors::CannotExecuteHelper :
        NotifyError (Exc.GetMsg(), 0, m_pWindow);
      return false;

      case EPIPE :
        NotifyError ("Your privileged session has been terminated, please try "
                     "again.", 0, m_pWindow);
        SetNoRoot();
      return false;

      case EOPNOTSUPP :
        NotifyError (Exc.GetMsg() + "\nOperation not supported by the driver.",
                     0, m_pWindow);
      break;

      case ENETDOWN :
        NotifyError ("Failed : the interface is currently down. Bring it up and"
                                                " try again.", 0, m_pWindow);
        IfaceIsDown();
      return false;

      case EPERM :
        SetNoRoot();
      // No break.

      case EAGAIN : case EBUSY : case EINVAL : case ENOSYS : case ENOTCONN :
      case nsErrors::InvalidData : case nsErrors::InvalidRootPassword :
      case ENOENT :
          NotifyError (std::string ("An error occured :\n") + Exc.GetMsg(),
                       Exc.GetCode(), m_pWindow);
      return false; // true on critical error, the program will exit.

      case nsErrors::OutOfMemory : return true;

      default :
        NotifyError (std::string ("Critical error :\n") + Exc.GetMsg(),
                     Exc.GetCode(), m_pWindow);
        m_Exc = Exc;
        if (::gtk_main_level()) ::gtk_main_quit();
    }
    return true;

} // ProcessError()


bool nsGUI::CGtkGUI::ApplyProfile (unsigned ProfileI) throw()
{
#ifndef NDEBUG
    std::cerr << "Applying a profile..." << std::endl;
#endif // NDEBUG
    try
    {
        if (BecomeRoot())
        {
            try
            {
                m_pModule->ApplyProfile (ProfileI);
            }
            catch (const nsErrors::CException& Exc)
            {
                if (Exc.GetCode() == ENETDOWN && BringIfaceUp())
                    m_pModule->ApplyProfile (ProfileI);
                else
                    throw;
            }
            SetIPFromProfile (ProfileI);
            ::g_timeout_add (800, reinterpret_cast< ::gboolean (*) (void*)>
                                        (RefreshRateDelayedCallBack), this);
            m_ToolTipHeader.clear();
            return true; // Select the row.
        }
    }
    catch (const nsErrors::CException& Exc)
    {
        ProcessError (Exc);
    }
    return false; // Don't select the row.

} // ApplyProfile()


bool nsGUI::CGtkGUI::DeleteProfile (unsigned ProfileI) throw()
{
    ::GtkDialog* const pDialog (GTK_DIALOG (::gtk_message_dialog_new (
                         m_pWindow, ::GtkDialogFlags (0), GTK_MESSAGE_QUESTION,
                         GTK_BUTTONS_YES_NO, "Do you want to delete this "
                         "profile : %s?", m_pModule->GetProfile
                         (m_ProfileTab.GetSelectedRow()).GetName().c_str())));
    const int Response (::gtk_dialog_run (pDialog));
    ::gtk_widget_destroy (GTK_WIDGET (pDialog));
    if (Response == GTK_RESPONSE_YES)
    {
        m_pModule->DeleteProfile (ProfileI);
        StoreProfiles();
        return true; // Delete the profile from the list.
    }
    return false; // Don't delete the profile from the list.

} // DeleteProfile()


void nsGUI::CGtkGUI::SelectMatchingProfile (const nsWireless::CCell& Cell)
                                                                        throw()
{
    for (unsigned i (m_pModule->GetNbProfile()) ; i-- ; )
        m_ProfileTab.SetSelected (i, m_pModule->GetProfile (i).Match (Cell));

} // SelectMatchingProfile()


bool nsGUI::CGtkGUI::ChangeModule (const std::string& IfaceName)
                                                    throw (nsErrors::CException)
{
#ifndef NDEBUG
    std::cerr << "CGtkGUI::ChangeModule()\n";
#endif // NDEBUG
    std::vector<nsWireless::CDeviceInfo> DevicesVec;
    nsCore::GetWirelessDevice (DevicesVec);
    if (DevicesVec.empty())
        throw nsErrors::CException ("Can't find any wireless network"
                                    " interface.", nsErrors::NoDeviceFound);
    int Pos (0);
    if (IfaceName.empty())
    {
        if (DevicesVec.size() > 1 || m_pModule.get())
            Pos = SelectInterface (GTK_WIDGET_VISIBLE (GTK_WIDGET (m_pWindow))
                                   ? m_pWindow : 0,
                                   DevicesVec);
    }
    else
    {
        for ( ; unsigned (Pos) < DevicesVec.size() &&
                        DevicesVec [Pos].GetDeviceName() != IfaceName ; ++Pos);
        if (unsigned (Pos) == DevicesVec.size())
            throw nsErrors::CException (std::string ("Can't find interface : ")
                                        + IfaceName, nsErrors::NoDeviceFound);
    }
    if (Pos == -1 || m_pModule.get() &&
                                    m_pModule->GetDriver().GetDeviceName() ==
                                            DevicesVec [Pos].GetDeviceName())
        return false;
    nsCore::IModule* pModule (0);
#ifndef NDEBUG
    std::cerr << "Instanciating module...\n";
#endif // NDEBUG
    try{pModule = nsCore::MakeModule (DevicesVec [Pos], this);}
    catch (const std::bad_alloc& Exc)
    {
        throw nsErrors::CException (Exc.what(), nsErrors::OutOfMemory);
    }
    if (!pModule)
        return false;
    if (m_GlobalTimer)
        ::g_source_remove (m_GlobalTimer);
    m_pModule.reset (pModule);
    try
    {
        if (IsRoot() && !IsReallyRoot())
            m_pModule->BecomeRoot();
#ifndef NDEBUG
        std::cerr << "Instanciating interface...\n";
#endif // NDEBUG
        m_pIface.reset (new nsNet::CInterface
                                    (m_pModule->GetDriver().GetDeviceName()));
    }
    catch (const std::bad_alloc&)
    {
        throw nsErrors::CException ("Out of memory.", nsErrors::OutOfMemory);
    }
    const bool IsIfaceUp (m_pIface->IsUp());
#ifndef NDEBUG
    std::cerr << "Interface is initially : " << (IsIfaceUp ? IfaceStatusLabelUp
                                        : IfaceStatusLabelDown) << std::endl;
#endif // NDEBUG
    IsIgnored (true);
    m_PrefsPage.IfaceStatus (IsIfaceUp);
    m_PrefsPage.IfaceLabel (IsIfaceUp ? IfaceStatusLabelUp
                                      : IfaceStatusLabelDown);
    IsIgnored (false);

    m_ProfileTab.Clear();
    for (unsigned i (0) ; i < m_pModule->GetNbProfile() ; ++i)
        m_ProfileTab.AddProfile (m_pModule->GetProfile (i));
    ::gtk_window_set_title (m_pWindow,
                 (m_pModule->GetDriver().GetProtoName() + " - RutilT").c_str());
    ::gtk_notebook_remove_page (m_pNoteBook, Private);
    if (m_pModule->GetPrivateTab())
    {
        ::gtk_notebook_append_page (m_pNoteBook, m_pModule->GetPrivateTab(),
                                    ::gtk_label_new (
                                m_pModule->GetDriver().GetProtoName().c_str()));
        ::gtk_widget_show_all (m_pModule->GetPrivateTab());
    }
    m_CellVec.clear();
    if (IsIfaceUp) // Limits possible errors.
        RefreshCellListCallBack (this);
#ifndef NDEBUG
    if (m_Exc.GetCode())
        std::cerr << "Error detected after fetching scan results : "
                  << m_Exc.GetMsg() << "\tCode : " << m_Exc.GetCode()
                  << std::endl;
#endif // NDEBUG
    RefreshRates();
    if (m_PrefsPage.DisplayTrayIcon())
    {
        m_GlobalTimer = ::g_timeout_add (GlobalTimerPeriod,
                                     reinterpret_cast< ::gboolean (*) (void*)>
                                        (GlobalTimerWithTrayCallBack), this);
        GlobalTimerWithTrayCallBack (this);
    }
    else
    {
        m_GlobalTimer = ::g_timeout_add (GlobalTimerPeriod,
                                     reinterpret_cast< ::gboolean (*) (void*)>
                                    (GlobalTimerWithoutTrayCallBack), this);
        GlobalTimerWithoutTrayCallBack (this);
    }
#ifndef NDEBUG
    std::cerr << "Exiting CGtkGUI::ChangeModule()\n";
#endif // NDEBUG
    return true;

} // ChangeModule()


bool nsGUI::CGtkGUI::NewProfile (unsigned CellI) throw()
{
    nsWireless::CCell& Cell (m_CellVec [CellI]);
    bool IsCurrentCell (false);
    try
    {
        if (m_pModule->GetDriver().GetAPMacAddr() == Cell.GetAPMacAddress())
            IsCurrentCell = true;
    }
    catch (const nsErrors::CSystemExc& Exc) {} // Ignored.
    const nsUserData::CProfile* const pProfile
                                     (m_pModule->NewProfile (m_pWindow, Cell));
    if (pProfile) // The user may change his mind and "cancel".
    {
        m_ProfileTab.AddProfile (*pProfile);
            // Check if the user hasn't modified the profile.
        if (IsCurrentCell && pProfile->Match (Cell))
            m_ProfileTab.SetSelected (m_pModule->GetNbProfile() - 1, true);
        return true;
    }
    return false;

} // NewProfile()


void nsGUI::CGtkGUI::SelectCurrentCell () throw (nsErrors::CSystemExc)
{
    nsWireless::CMacAddress Addr;
    try
    {
        Addr = m_pModule->GetDriver().GetAPMacAddr();
    }
    catch (const nsErrors::CSystemExc& Exc)
    {
        if (Exc.GetCode() != ENETDOWN)
            throw;
    }
    SelectCurrentCell (Addr);

} // SelectCurrentCell()


void nsGUI::CGtkGUI::SelectCurrentCell (const nsWireless::CMacAddress& Addr)
                                                                        throw()
{
    for (unsigned i (m_CellVec.size()) ; i-- ; )
        m_SiteSurveyTab.SetSelected (i,
                                     m_CellVec [i].GetAPMacAddress() == Addr);

} // SelectCurrentCell()


bool nsGUI::CGtkGUI::BecomeRoot () throw (nsErrors::CException)
{
    if (!IsRoot())
    {
#ifdef NHELPER
        ::GtkWidget* const pDialog = ::gtk_message_dialog_new
                                            (m_pWindow,
                                             GTK_DIALOG_DESTROY_WITH_PARENT,
                                             GTK_MESSAGE_ERROR,
                                             GTK_BUTTONS_CLOSE,
                                             "RutilT must be ran as root to"
                                             " do that.");
        ::gtk_dialog_run (GTK_DIALOG (pDialog));
        ::gtk_widget_destroy (pDialog);
#else
#ifdef NOROOTPASSCHECK
        if (nsRoot::CSu::Init (""))
        {
            m_pModule->BecomeRoot();
            m_pIface.reset (new nsNet::CInterfaceRoot (*m_pIface));
            SetRoot();
            IsIgnored (true);
            m_PrefsPage.SetRootMode (IsRoot());
            IsIgnored (false);
            AutoLeaveRootModeCallBack (this);
        }
        else
        {
            NotifyError ("An error occured.\nPlease try again.", 0, m_pWindow);
            SetNoRoot();
        }
#else 
        ::GtkDialog* const pDialog (GTK_DIALOG (::gtk_dialog_new_with_buttons
                            ("Enter password", m_pWindow,
                             ::GtkDialogFlags (GTK_DIALOG_MODAL |
                             GTK_DIALOG_NO_SEPARATOR), GTK_STOCK_CANCEL,
                             GTK_RESPONSE_CANCEL, GTK_STOCK_OK, GTK_RESPONSE_OK,
                             static_cast<void*> (0))));
        ::GtkContainer* const pHbox (GTK_CONTAINER
                                                (::gtk_hbox_new (false, 6)));
        ::gtk_container_add (pHbox, ::gtk_image_new_from_stock
                    (GTK_STOCK_DIALOG_AUTHENTICATION, GTK_ICON_SIZE_DIALOG));
        ::gtk_container_add (pHbox, ::gtk_label_new ("You need administrator"
                 " privileges to do this,\nplease enter your root password."));
        ::gtk_container_add (GTK_CONTAINER (pDialog->vbox),
                             GTK_WIDGET (pHbox));
        ::GtkEntry* const pEntry (GTK_ENTRY (::gtk_entry_new()));
        ::gtk_entry_set_visibility (pEntry, false);
        ::g_signal_connect (G_OBJECT (pEntry), "activate",
                            G_CALLBACK (TriggerResponseCallBack), pDialog);
        ::gtk_container_add (GTK_CONTAINER (pDialog->vbox),
                             GTK_WIDGET (pEntry));
        ::gtk_container_set_border_width (GTK_CONTAINER (pDialog), 6);
        ::gtk_window_set_resizable (GTK_WINDOW (pDialog), false);
        ::gtk_widget_show_all (GTK_WIDGET (pDialog));

        try
        {
            bool IsMsgDisplayed (false);
            while (!IsRoot() && ::gtk_dialog_run (pDialog) == GTK_RESPONSE_OK)
                if (nsRoot::CSu::Init (::gtk_entry_get_text (pEntry)))
                {
                    m_pModule->BecomeRoot();
                    m_pIface.reset (new nsNet::CInterfaceRoot (*m_pIface));
                    SetRoot();
                    AutoLeaveRootModeCallBack (this);
                }
                else
                {
                    ::gtk_entry_set_text (pEntry, "");
                    if (!IsMsgDisplayed)
                    {
                        ::GtkWidget* const pLabel (::gtk_label_new
                                    ("Invalid password, please try again."));
                        ::gtk_box_pack_start_defaults (GTK_BOX (pDialog->vbox),
                                                       pLabel);
                        ::gtk_widget_show (pLabel);
                        IsMsgDisplayed = true;
                    }
                }
        }
        catch (const std::bad_alloc& Exc)
        {
            ::gtk_widget_destroy (GTK_WIDGET (pDialog));
            IsIgnored (true);
            m_PrefsPage.SetRootMode (IsRoot());
            IsIgnored (false);
            throw nsErrors::CSystemExc (Exc.what(), nsErrors::OutOfMemory);
        }
        catch (const nsErrors::CException& Exc)
        {
            ::gtk_widget_destroy (GTK_WIDGET (pDialog));
            IsIgnored (true);
            m_PrefsPage.SetRootMode (IsRoot());
            IsIgnored (false);
            throw;
        }
        IsIgnored (true);
        m_PrefsPage.SetRootMode (IsRoot());
        IsIgnored (false);
        ::gtk_widget_destroy (GTK_WIDGET (pDialog));
#endif // NOROOTPASSCHECK
#endif // NHELPER
    }
    return IsRoot();

} // BecomeRoot()


bool nsGUI::CGtkGUI::RefreshStats () throw()
{
    try
    {
        m_pIface->RefreshStats();
        m_StatsPage.SetTXBytes (m_pIface->GetTXBytes());
        m_StatsPage.SetTXPackets (m_pIface->GetTXPackets());
        m_StatsPage.SetTXErrors (m_pIface->GetTXErrors());
        m_StatsPage.SetCollisions (m_pIface->GetCollisions());
        m_StatsPage.SetRXBytes (m_pIface->GetRXBytes());
        m_StatsPage.SetRXPackets (m_pIface->GetRXPackets());
        m_StatsPage.SetRXErrors (m_pIface->GetRXErrors());
    }
    catch (const nsErrors::CException& Exc)
    {
        if (Exc.GetCode() == nsErrors::StatsParsing)
            NotifyError ("Statistics unavailable.", 0, m_pWindow);
        else NotifyError (Exc.GetMsg(), Exc.GetCode(), m_pWindow);
        return false;
    }
    return true;

} // RefreshStats()


void nsGUI::CGtkGUI::SetIPFromProfile (unsigned ProfileI)
                                                    throw (nsErrors::CException)
{
    const nsUserData::CProfile& Profile (m_pModule->GetProfile (ProfileI));
    if (Profile.GetIPSettings().GetSetting() == nsUserData::Dhcp)
    {
        m_pIface->SetIPFromDHCP (Profile.GetSSID());
        IsIgnored (true);
        m_PrefsPage.IfaceStatus (true);
        m_PrefsPage.IfaceLabel (IfaceStatusLabelUp);
        IsIgnored (false);
    }

} // SetIPFromProfile()


void nsGUI::CGtkGUI::RefreshLinkStatus () throw (nsErrors::CException)
{
    nsWireless::CQuality Quality;
    unsigned TxRate (0);
    try
    {
        Quality = m_pModule->GetDriver().GetQuality();
    }
    catch (...) {}
    try
    {
        TxRate = m_pModule->GetDriver().GetTxRate();
    }
    catch (...) {}
    nsWireless::CMacAddress Addr;
    try
    {
        Addr = m_pModule->GetDriver().GetAPMacAddr();
    }
    catch (...) {}
    RefreshLinkStatus (Quality, TxRate, Addr);

} // RefreshLinkStatus()


void nsGUI::CGtkGUI::RefreshLinkStatus (const nsWireless::CQuality& Quality,
                    unsigned TxRate, const nsWireless::CMacAddress& APMacAddr)
                                                throw (nsErrors::CException)
{
    try
    {
        m_LinkStatusTab.SetChannel (m_pModule->GetDriver().GetChannel(),
                                    m_pModule->GetDriver().GetFrequency());
        m_LinkStatusTab.SetMode (nsWireless::GetModeName
                                            (m_pModule->GetDriver().GetMode()));
    }
    catch (const nsErrors::CSystemExc& Exc)
    {
        if (Exc.GetCode() != ENETDOWN)
            throw;
        IfaceIsDown();
        m_LinkStatusTab.SetChannel (0, 0.);
        m_LinkStatusTab.SetMode ("N/A");
    }
    if (!APMacAddr.Empty())
    {
        m_LinkStatusTab.SetStatus (m_pModule->GetDriver().GetSSID(),
                                   APMacAddr.GetStr());
        m_LinkStatusTab.SetLinkQuality (Quality.GetLinkQuality());
        m_LinkStatusTab.SetSignalLevel (Quality.GetSignalLevel());
        m_LinkStatusTab.SetNoiseLevel (Quality.GetNoiseLevel());
        m_LinkStatusTab.SetTxRate (TxRate);
    }
    else m_LinkStatusTab.Clear();
    try{m_LinkStatusTab.SetAddr (m_pIface->GetIP().GetStr());}
    catch (const nsErrors::CSystemExc& Exc)
    {
        if (Exc.GetCode() == EADDRNOTAVAIL)
            m_LinkStatusTab.SetAddr ("N/A");
        else throw;
    }

} // RefreshLinkStatus()


void nsGUI::CGtkGUI::TrayIconClickedCallBack (CGtkGUI* This,
                                              ::GdkEventButton* pEvent) throw()
{
    if (pEvent->button == 3) // Right button.
        This->TrayIconMenu (*pEvent);
    else if (GTK_WIDGET_VISIBLE (GTK_WIDGET (This->m_pWindow)))
        HideCallBack (This);
    else
        ShowCallBack (This);

} // TrayIconClickedCallBack()


void nsGUI::CGtkGUI::RefreshTray (const nsWireless::CQuality& Quality,
                                  unsigned TxRate) throw()
{
    unsigned Average (Quality.GetLinkQuality());
    Average += unsigned ((Quality.GetSignalLevel() + 256U) / 256.0 * 100U);
    Average += unsigned (Quality.GetNoiseLevel() / 256.0 * -100);
    Average /= 3U;
    if (Average > 66U)
        m_TrayIcon.SetIcon (nsCore::TrayIconHighPath);
    else if (Average > 33U)
        m_TrayIcon.SetIcon (nsCore::TrayIconMediumPath);
    else
        m_TrayIcon.SetIcon (nsCore::TrayIconLowPath);

    std::ostringstream Os;
    Os << m_ToolTipHeader << '\n' << "Link quality : "
       << Quality.GetLinkQuality() << " %\n"
       << "Signal level : " << Quality.GetSignalLevel() << " dBm\n"
       << "Noise level : " << Quality.GetNoiseLevel() << " dBm\n"
          "Tx rate : ";
    if (TxRate)
        Os << TxRate / 1000.0 << " Mbps";
    else
        Os << "Unknown.";
    m_TrayIcon.SetTooltip (Os.str());

} // RefreshTray()


void nsGUI::CGtkGUI::RootModeChangedCallBack (CGtkGUI* This) throw()
{
    if (!This->IsIgnored())
    {
        if (This->IsRoot())
            This->SetNoRoot();
        else try
        {
            This->BecomeRoot();
        }
        catch (const nsErrors::CException& Exc)
        {
            if (!This->ProcessError (Exc))
                This->SetNoRoot();
        }
    }

} // RootModeChangedCallBack()


void nsGUI::CGtkGUI::ChangeInterfaceCallBack (CGtkGUI* This) throw()
{
    try {This->ChangeModule();}
    catch (const nsErrors::CException& Exc) {This->ProcessError (Exc);}

} // ChangeInterfaceCallBack()


void nsGUI::CGtkGUI::AutoLeaveRootModeCallBack (CGtkGUI* This) throw()
{
    if (!This->IsIgnored())
    {
        if (This->IsRoot())
        {
            if (This->m_LeaveRootModeTimer)
            {
                ::g_source_remove (This->m_LeaveRootModeTimer);
                This->m_LeaveRootModeTimer = 0;
            }
            const unsigned Delay (This->m_PrefsPage.GetAutoLeaveRootDelay() *
                                                                        1000);
            if (Delay)
                This->m_LeaveRootModeTimer = ::g_timeout_add (Delay,
                                     reinterpret_cast< ::gboolean (*) (void*)>
                                            (LeaveRootModeTimerCallBack), This);
        }
        This->StoreOptions();
    }

} // AutoLeaveRootModeCallBack()


bool nsGUI::CGtkGUI::LeaveRootModeTimerCallBack (CGtkGUI* This) throw()
{
    This->SetNoRoot();
    return false; // Stop the timer.

} // LeaveRootModeTimerCallBack()


void nsGUI::CGtkGUI::RateChangedCallBack (CGtkGUI* This) throw()
{
#ifndef NDEBUG
    std::cerr << "Entering CGtkGUI::RateChangedCallBack(), ignoring : "
              << std::boolalpha << This->IsIgnored() << std::noboolalpha
              << std::endl;
#endif // NDEBUG
    if (!This->IsIgnored())
    {
        try
        {
            if (This->BecomeRoot())
            {
                This->m_pModule->GetDriver().SetTxRate
                                        (This->m_PrefsPage.GetSelectedRate());
                This->m_CurrentRateIndex =
                                    This->m_PrefsPage.GetSelectedRateIndex();
                return;
            }
        }
        catch (const nsErrors::CException& Exc)
        {
            if (This->ProcessError (Exc))
                return;
        }   // Revert to previous state (in case of error, not root etc...).
        This->IsIgnored (true);
        This->m_PrefsPage.SetRateIndex (This->m_CurrentRateIndex);
        This->IsIgnored (false);
    }
#ifndef NDEBUG
    std::cerr << "Leaving CGtkGUI::RateChangedCallBack()" << std::endl;
#endif // NDEBUG

} // RateChangedCallBack()


void nsGUI::CGtkGUI::SetNoRoot () throw()
{
    m_Flags &= ~IsRootF;
    nsRoot::CSu::Close();
    IsIgnored (true);
    m_PrefsPage.SetRootMode (false);
    IsIgnored (false);
    if (m_LeaveRootModeTimer)
    {
        ::g_source_remove (m_LeaveRootModeTimer);
        m_LeaveRootModeTimer = 0;
    }

} // SetNoRoot()


void nsGUI::CGtkGUI::RefreshRates () throw()
{
#ifndef NDEBUG
    std::cerr << "CGtkGUI::RefreshRates()\n";
#endif // NDEBUG
    std::vector<int> RatesVec;
    try {m_pModule->GetDriver().GetSupportedRates (RatesVec);}
    catch (const nsErrors::CException& Exc)
    {
#ifndef NDEBUG
        std::cerr << "Cannot get supported rates : " << Exc.GetMsg() << ' '
                  << Exc.GetCode() << std::endl;
#endif // NDEBUG
    }
    IsIgnored (true);
    m_PrefsPage.SetRates (RatesVec);
    IsIgnored (false);
    if (!RatesVec.empty())
    {
        unsigned Index (0);
        try
        {
            Index = std::find (RatesVec.begin(), RatesVec.end(),
                               m_pModule->GetDriver().GetTxRate()) -
                                                            RatesVec.begin();
        }
        catch (const nsErrors::CException& Exc)
        {
#ifndef NDEBUG
            std::cerr << "Cannot get current TX rate : " << Exc.GetMsg() << ' '
                      << Exc.GetCode() << std::endl;
#endif // NDEBUG
        }
        IsIgnored (true);
        m_PrefsPage.SetRateIndex (Index);
        m_CurrentRateIndex = Index;
        IsIgnored (false);
    }
#ifndef NDEBUG
    std::cerr << "Exiting CGtkGUI::RefreshRates()\n";
#endif // NDEBUG

} // RefreshRates()


bool nsGUI::CGtkGUI::RefreshRateDelayedCallBack (CGtkGUI* This) throw()
{
    This->RefreshRates();
    return false;

} // RefreshRateDelayedCallBack()


void nsGUI::CGtkGUI::DisplayTrayIconButtonCallBack (CGtkGUI* This) throw()
{
    if (!This->IsIgnored())
    {
        ::g_signal_handler_disconnect (This->m_pWindow, This->m_ExitHandlerID);
        if (This->m_PrefsPage.DisplayTrayIcon())
        {
            This->m_TrayIcon.Show();
            ::g_source_remove (This->m_GlobalTimer);
            This->m_ExitHandlerID = ::g_signal_connect_swapped
                                    (G_OBJECT (This->m_pWindow), "delete_event",
                                            G_CALLBACK (HideCallBack), This);
            This->m_GlobalTimer = ::g_timeout_add (GlobalTimerPeriod,
                                     reinterpret_cast< ::gboolean (*) (void*)>
                                        (GlobalTimerWithTrayCallBack), This);
            GlobalTimerWithTrayCallBack (This);
        }
        else
        {
            This->m_TrayIcon.Hide();
            ::g_source_remove (This->m_GlobalTimer);
            This->m_ExitHandlerID = ::g_signal_connect (G_OBJECT
                                            (This->m_pWindow), "delete_event",
                                                G_CALLBACK (DeleteCallBack), 0);
            This->m_GlobalTimer = ::g_timeout_add (GlobalTimerPeriod,
                                     reinterpret_cast< ::gboolean (*) (void*)>
                                    (GlobalTimerWithoutTrayCallBack), This);
            GlobalTimerWithoutTrayCallBack (This);
        }
        This->StoreOptions();
    }

} // DisplayTrayIconButtonCallBack()


void nsGUI::CGtkGUI::StartMinimizedButtonCallBack (CGtkGUI* This) throw()
{
    if (!This->IsIgnored())
        This->StoreOptions();

} // StartMinimizedButtonCallBack()


void nsGUI::CGtkGUI::LoadOptions () throw()
{
        // The constructor parameters are the default settings.
    nsUserData::COptionsFile OptionsFile (true, false, 10);
    try{OptionsFile.Extract();}
    catch (...) {} // Ignore all errors.
    IsIgnored (true);
    m_PrefsPage.DisplayTrayIcon (OptionsFile.DisplayTrayIcon());
    m_PrefsPage.StartMinimized (OptionsFile.StartMinimized());
    m_PrefsPage.SetAutoLeaveRootDelay (OptionsFile.LeaveRootDelay());
    IsIgnored (false);

} // LoadOptions()


void nsGUI::CGtkGUI::StoreProfiles () throw()
{
    try
    {
        m_pModule->RecordProfiles();
    }
    catch (const nsErrors::CException& Exc)
    {
        NotifyError ("Profiles could not have been recorded.", Exc.GetCode(),
                                                                    m_pWindow);
    }

} // StoreProfiles()


void nsGUI::CGtkGUI::StoreOptions () throw()
{
    try
    {
        nsUserData::COptionsFile OptionsFile (m_PrefsPage.DisplayTrayIcon(),
                                              m_PrefsPage.StartMinimized(),
                                          m_PrefsPage.GetAutoLeaveRootDelay());
        OptionsFile.Record();
    }
    catch (const nsErrors::CException& Exc)
    {
        NotifyError ("Options could not have been recorded.", Exc.GetCode(),
                     m_pWindow);
    }

} // StoreOptions()


void nsGUI::CGtkGUI::IfaceStatusChangedCallBack (CGtkGUI* This) throw()
{
    if (!This->IsIgnored())
    {
        const bool BringDown (!This->m_PrefsPage.IfaceStatus());
        if (This->BecomeRoot())
            try
            {
                if (BringDown)
                {
                    This->m_pIface->Down();
                    This->m_PrefsPage.IfaceLabel (IfaceStatusLabelDown);
                }
                else
                {
                    This->m_pIface->Up();
                    This->m_PrefsPage.IfaceLabel (IfaceStatusLabelUp);
                }
                return;
            }
            catch (const nsErrors::CException& Exc)
            {
                if (This->ProcessError (Exc))
                    return;
            }
            // Revert to previous state (in case of error, not root etc...).
        This->IsIgnored (true);
        This->m_PrefsPage.IfaceStatus (BringDown);
        This->IsIgnored (false);
    }

} // IfaceStatusChangedCallBack()


bool nsGUI::CGtkGUI::BringIfaceUp () throw()
{
    bool B (AskUser ("The interface needs to be up to perform this"
                           " action.\nBring it up now?", m_pWindow));
    if (B)
        try
        {
            m_pIface->Up();
        }
        catch (const nsErrors::CSystemExc& Exc)
        {
            if (ProcessError (Exc))
                return false;
            B = false;
        }
    IsIgnored (true);
    m_PrefsPage.IfaceStatus (B);
    m_PrefsPage.IfaceLabel (B ? IfaceStatusLabelUp : IfaceStatusLabelDown);
    IsIgnored (false);
    return B;

} // BringIfaceUp()


void nsGUI::CGtkGUI::IfaceIsDown () throw()
{
    IsIgnored (true);
    m_PrefsPage.IfaceStatus (false);
    m_PrefsPage.IfaceLabel (IfaceStatusLabelDown);
    IsIgnored (false);

} // IfaceIsDown()


void nsGUI::CGtkGUI::TrayIconMenu (::GdkEventButton& Event) throw()
{
    if (m_pTrayMenu)
        ::gtk_object_destroy (GTK_OBJECT (m_pTrayMenu));
    m_pTrayMenu = GTK_MENU_SHELL (::gtk_menu_new());
    ::gtk_menu_shell_append (m_pTrayMenu, MakeMenuVisibilityItem());
    ::gtk_menu_shell_append (m_pTrayMenu, MakeMenuProfileItem());
    ::gtk_menu_shell_append (m_pTrayMenu, ::gtk_separator_menu_item_new());
    ::gtk_menu_shell_append (m_pTrayMenu, MakeMenuExitItem());
    ::gtk_widget_show_all (GTK_WIDGET (m_pTrayMenu));
    ::gtk_menu_popup (GTK_MENU (m_pTrayMenu), 0, 0, 0, 0, Event.button,
                                                                    Event.time);

} // TrayIconMenu()


::GtkWidget* nsGUI::CGtkGUI::MakeMenuVisibilityItem () throw()
{
    ::GtkWidget* pItem;
    if (GTK_WIDGET_VISIBLE (GTK_WIDGET (m_pWindow)))
    {
        pItem = ::gtk_menu_item_new_with_label ("Hide window");
        ::g_signal_connect_swapped (pItem, "activate",
                                            G_CALLBACK (HideCallBack), this);
    }
    else
    {
        pItem = ::gtk_menu_item_new_with_label ("Show window");
        ::g_signal_connect_swapped (pItem, "activate",
                                            G_CALLBACK (ShowCallBack), this);
    }
    return pItem;

} // MakeMenuVisibilityItem()


::GtkWidget* nsGUI::CGtkGUI::MakeMenuProfileItem () throw()
{
    ::GtkWidget* const pItem (::gtk_menu_item_new_with_label ("Apply profile"));
    ::GtkMenuShell* const pSubMenu (GTK_MENU_SHELL (::gtk_menu_new()));
    ::gtk_menu_item_set_submenu (GTK_MENU_ITEM (pItem), GTK_WIDGET (pSubMenu));
    const ::gint NProfile (::gint (m_pModule->GetNbProfile()));
    if (NProfile)
    {
        for (::gint I (0) ; I < NProfile ; ++I)
        {
            ::GtkWidget* const pProfileItem (::gtk_menu_item_new_with_label
                                (m_pModule->GetProfile (I).GetName().c_str()));
            ::gtk_menu_shell_append (pSubMenu, pProfileItem);
            ::g_signal_connect_swapped (pProfileItem, "activate",
                    G_CALLBACK (ApplyProfileTrayCallBack), GINT_TO_POINTER (I));
        }
        ::gtk_menu_shell_append (pSubMenu, ::gtk_separator_menu_item_new());
    }
    ::GtkWidget* const pNewProfileItem (::gtk_image_menu_item_new_from_stock
                                                            (GTK_STOCK_NEW, 0));
    ::g_signal_connect_swapped (pNewProfileItem, "activate",
                            G_CALLBACK (CreateProfileAndApplyCallBack), this);
    ::gtk_menu_shell_append (pSubMenu, pNewProfileItem);
    return pItem;

} // MakeMenuProfileItem()


::GtkWidget* nsGUI::CGtkGUI::MakeMenuExitItem() throw()
{
    ::GtkWidget* const pItem (::gtk_image_menu_item_new_from_stock
                                                        (GTK_STOCK_QUIT, 0));
        // The signature of DeleteCallBack() does not quite match.
    ::g_signal_connect (pItem, "activate", G_CALLBACK (DeleteCallBack), 0);
    return pItem;

} // MakeMenuExitItem()


void nsGUI::CGtkGUI::HideCallBack (CGtkGUI* This) throw()
{
    ::gtk_widget_hide (GTK_WIDGET (This->m_pWindow));

} // HideCallBack()


void nsGUI::CGtkGUI::ShowCallBack (CGtkGUI* This) throw()
{
    ::gtk_widget_show_all (GTK_WIDGET (This->m_pWindow));
    try
    {       // WARNING The NoteBookPage is NOT passed.
        PageSwitchCallBack (This->m_pNoteBook, 0,
                            PageNum_e (::gtk_notebook_get_current_page
                                                (This->m_pNoteBook)),
                            This);
    }
    catch (const nsErrors::CException& Exc) {This->ProcessError (Exc);}

} // ShowCallBack()


void nsGUI::CGtkGUI::CreateProfileAndApplyCallBack (CGtkGUI* This) throw()
{
    const unsigned LastProfileI (This->m_pModule->GetNbProfile());
    const nsUserData::CProfile* pProfile (This->NewProfile());
    if (pProfile)
    {
        This->m_ProfileTab.AddProfile (*pProfile);
        This->m_ProfileTab.SetSelected (LastProfileI,
                                            This->ApplyProfile (LastProfileI));
        This->SelectMatchingProfile (LastProfileI);
    }

} // CreateProfileAndApplyCallBack()


void nsGUI::CGtkGUI::SelectMatchingProfile (unsigned ProfileI) throw()
{
    for (unsigned I (m_pModule->GetNbProfile()) ; I-- ; )
            // FIXME Could check if profiles are "equals".
        m_ProfileTab.SetSelected (I, I == ProfileI);

} // ProfileI()


void nsGUI::CGtkGUI::ApplyProfileTrayCallBack (::gpointer ProfileIPtr) throw()
{
    const ::gint ProfileI (GPOINTER_TO_INT (ProfileIPtr));
    if (GlobalUglyThis->ApplyProfile (ProfileI))
        GlobalUglyThis->SelectMatchingProfile (ProfileI);

} // ApplyProfileTrayCallBack()


const nsUserData::CProfile* nsGUI::CGtkGUI::EditProfile () throw()
{
    if (m_pModule->EditProfile (m_pWindow, m_ProfileTab.GetSelectedRow()))
    {
        StoreProfiles();
        return &m_pModule->GetProfile (m_ProfileTab.GetSelectedRow());
    }
    return 0;

} // EditProfile()


const nsUserData::CProfile* nsGUI::CGtkGUI::NewProfile () throw()
{
    const nsUserData::CProfile* const pProfile (m_pModule->NewProfile
                                                                (m_pWindow));
    if (pProfile)
        StoreProfiles();
    return pProfile;

} // NewProfile()
