/*****************************************************************************
 * kiwix.h - QStarDict, a dictionary for learning foreign languages          *
 * Copyright (C) 2023 Alexander Rodin                                        *
 *                                                                           *
 * This program is free software; you can redistribute it and/or modify      *
 * it under the terms of the GNU General Public License as published by      *
 * the Free Software Foundation; either version 2 of the License, or         *
 * (at your option) any later version.                                       *
 *                                                                           *
 * This program is distributed in the hope that it will be useful,           *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of            *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             *
 * GNU General Public License for more details.                              *
 *                                                                           *
 * You should have received a copy of the GNU General Public License along   *
 * with this program; if not, write to the Free Software Foundation, Inc.,   *
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.               *
 *****************************************************************************/

#ifndef KIWIX_H
#define KIWIX_H

#include "../dictplugin.h"
#include "../metadata.h"

#include <string>
#include <QDir>
#include <QHash>
#include <QVector>
#include <zim/archive.h>

class Kiwix: public QObject, public QStarDict::BasePlugin, public QStarDict::DictPlugin, public QStarDict::ConfigurablePlugin
{
	Q_OBJECT
	Q_PLUGIN_METADATA(IID "org.qstardict.KiwixPlugin/1.0" FILE "kiwix.json")
	Q_INTERFACES(QStarDict::BasePlugin QStarDict::DictPlugin QStarDict::ConfigurablePlugin)

	public:
		Kiwix(QObject *parent = 0);
		~Kiwix();

        QIcon pluginIcon() const;
		Features features() const
		{ return Features(Feature::SearchSimilar); }

		QStringList availableDicts() const;
		QStringList loadedDicts() const;
		void setLoadedDicts(const QStringList &loadedDicts);
		DictInfo dictInfo(const QString &dict);

		bool isTranslatable(const QString &dict, const QString &word);
		Translation translate(const QString &dict, const QString &word);
		QStringList findSimilarWords(const QString &dict, const QString &word);

        std::optional<QString> dictionaryFileFilter() const
        { return tr("Kiwix ZIM files (*.zim)"); }

        std::optional<QString> addDictionary(QWidget *parent, const QString &fileName);
        bool isDictionaryRemovable(const QString &dict)
        { return true; /* all of the dictionaries are stored in dataDirPath(), so all of them can be removed */ }
        bool removeDictionary(const QString &dict);

        static QString dataDirPath()
        { return QDir::homePath() + "/.qstardict/dic/kiwix"; }

        int execSettingsDialog(QWidget *parent = nullptr) override;

        /**
         * Whether "remove colors" option is enabled
         */
        bool removeColors() const
        { return m_removeColors; }

        /**
         * Set "remove colors" option
         */
        void setRemoveColors(bool removeColors)
        { m_removeColors = removeColors; }

	private:
        QHash<QString, zim::Archive*> m_archives;
        bool m_removeColors;

        void saveSettings();
};

#endif // KIWIX_H

// vim: tabstop=4 softtabstop=4 shiftwidth=4 expandtab cindent

