/*
 * Control and monitor features on logitech dual optical mice.
 * Requires libusb.
 * 
 * Copyright (C) Brad Hards, 2002, 2003, 2004
 */

/*  
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

#include <stdio.h>
#ifdef HAVE_STRING_H
#include <string.h>
#endif
#include <getopt.h>
#ifdef HAVE_STDINT_H
#include <stdint.h>
#endif
#ifdef HAVE_INTTYPES_H
#include <inttypes.h>
#endif
#include <usb.h>

#ifndef USB_ENDPOINT_IN
#define USB_ENDPOINT_IN 0x80
#endif

#define VENDOR_LOGITECH 0x046D

int get_resolution(struct usb_device *dev)
{
    char resolution;
    int result;
    usb_dev_handle *usb_h;

    usb_h = usb_open(dev);
    if (0 > usb_h) {
	printf("Error opening usbfs file: %s\n", usb_strerror());
	return -1;
    }

    result =  usb_control_msg(usb_h,
			      USB_TYPE_VENDOR | USB_ENDPOINT_IN,
			      0x01,
			      0x000E,
			      0x0000,
			      &resolution,
			      0x0001,
			      100);

    if (0 > result) {
	printf("Error getting resolution from device : %s\n", usb_strerror());
    }

    usb_close(usb_h);

    return resolution;
}

/* 
 * this is a bit subtle. The use_channel2 gets set to 0x0100 if we
 * need to use the second channel, and 0x0000 otherwise.
 * This just happens to match up with setting the fourth and 
 * sixth bytes to 0x01, with a simple bit-wise OR.
 */
int get_csr(struct usb_device *dev,  int use_channel2, uint8_t *status)
{
    int result = 0;
    usb_dev_handle *usb_h;

    usb_h = usb_open(dev);
    if (0 > usb_h) {
	printf("Error opening usbfs file: %s\n", usb_strerror());
	return -1;
    }

    result =  usb_control_msg(usb_h,
			      USB_TYPE_VENDOR | USB_ENDPOINT_IN,
			      0x09,
			      0x0003,
			      (0x0000 | use_channel2),
			      status,
			      (0x0008 | use_channel2),
			      1000);

    if (0 > result) {
	printf("   Error getting cordless status from device : %s\n", usb_strerror());
    }

    usb_close(usb_h);

    return result;
}

/* 
 * this is a bit subtle. The use_channel2 gets set to 0x0100 if we
 * need to use the second channel, and 0x0000 otherwise.
 * This just happens to match up with setting the fourth and 
 * sixth bytes to 0x01, with a simple bit-wise OR.
 */
int get_cruise(struct usb_device *dev, int use_channel2)
{
    char cruise = 0;
    int result = 0;
    usb_dev_handle *usb_h;

    usb_h = usb_open(dev);
    if (0 > usb_h) {
	printf("Error opening usbfs file: %s\n", usb_strerror());
	return -1;
    }

    result =  usb_control_msg(usb_h,
			      USB_TYPE_VENDOR | USB_ENDPOINT_IN,
			      0x01,
			      0x17,
			      (0x0000 | use_channel2),
			      &cruise,
			      (0x0001 | use_channel2),
			      1000);
    if (0 > result) {
	printf("   Error getting cruise control setting from device : %s\n", usb_strerror());
    }

    usb_close(usb_h);

    return cruise;
}

/* resolution should be 0x03 for 400cpi, 0x04 for 800cpi */
int set_resolution(struct usb_device *dev, int resolution)
{
    usb_dev_handle *usb_h;
    int result = 0;

    usb_h = usb_open(dev);
    if (0 > usb_h) {
	printf("Error opening usbfs file: %s\n", usb_strerror());
	return -1;
    }

    usb_control_msg(usb_h, USB_TYPE_VENDOR, 0x02, 0x000E, resolution,
		    NULL,  0x0000, 100);
    if (0 > result) {
	printf("Error setting resolution on device : %s\n", usb_strerror());
    }

    usb_close(usb_h);

    return 0;
}

int enable_cruise(struct usb_device *dev)
{
    usb_dev_handle *usb_h;
    int result = 0;

    usb_h = usb_open(dev);
    if (0 > usb_h) {
	printf("Error opening usbfs file: %s\n", usb_strerror());
	return -1;
    }

    usb_control_msg(usb_h, USB_TYPE_VENDOR, 0x02, 0x0017, 0x01,
		    NULL,  0x0000, 100);
    if (0 > result) {
	printf("Error enabling cruise control on device : %s\n", usb_strerror());
    }

    usb_close(usb_h);

    return 0;
}

int disable_cruise(struct usb_device *dev)
{
    usb_dev_handle *usb_h;
    int result = 0;

    usb_h = usb_open(dev);
    if (0 > usb_h) {
	printf("Error opening usbfs file: %s\n", usb_strerror());
	return -1;
    }

    usb_control_msg(usb_h, USB_TYPE_VENDOR, 0x02, 0x0017, 0x00,
		    NULL,  0x0000, 100);
    if (0 > result) {
	printf("Error disabling cruise control on device : %s\n", usb_strerror());
    }

    usb_close(usb_h);

    return 0;
}

void usage(void)
{
    printf("Logitech Mouse Applet, Version %s\n", VERSION);
    printf(" --get-res     read the current sensor resolution\n");
    printf(" --set-res=X   set the sensor resolution to X, where X is 400 or 800\n");
    printf(" --get-cc      read the current Cruise Control setting\n");
    printf(" --enable-cc   enable Cruise Control\n");
    printf(" --disable-cc  disable Cruise Control\n");
    printf(" --version     display the program version\n");
    printf(" --help        print this usage information\n");
    printf(" -c            same as --get-cc\n");
    printf(" -d            same as --disable-cc\n");
    printf(" -e            same as --enable-cc\n");
    printf(" -g            same as --get-res\n");
    printf(" -s X          same as --set-res X\n");
    printf(" -h            same as --help\n");
    printf(" -v            same as --version\n");
    printf("Copyright (C) 2002-2004 Brad Hards <bradh@frogmouth.net>\n");
}

void version(void)
{
    printf("Logitech Mouse Applet, Version %s\n", VERSION);
}

#define HAS_RES 0x01  /* mouse supports variable resolution */
#define HAS_SS  0x02  /* mouse supports smart scroll control */
#define HAS_CSR  0x04  /* mouse supports cordless status reporting and control */
#define HAS_SSR  0x08  /* mouse supports smart scroll reporting */
#define USE_CH2  0x10 /* mouse needs to use the second channel */

struct device_table {
    int idVendor;
    int idProduct;
    char* Model;
    char* Name;
    int flags;
} device_table[] = {
    { VENDOR_LOGITECH, 0xC00E, "M-BJ58", "Wheel Mouse Optical", HAS_RES },
    { VENDOR_LOGITECH, 0xC00F, "M-BJ79", "MouseMan Traveler", HAS_RES },
    { VENDOR_LOGITECH, 0xC012, "M-BL63B", "MouseMan Dual Optical", HAS_RES },
    { VENDOR_LOGITECH, 0xC01B, "M-BP86", "MX310 Optical Mouse", HAS_RES },
    { VENDOR_LOGITECH, 0xC01D, "M-BS81A", "MX510 Optical Mouse", HAS_RES | HAS_SS | HAS_SSR },
    { VENDOR_LOGITECH, 0xC024, "M-BP82", "MX300 Optical Mouse", HAS_RES },
    { VENDOR_LOGITECH, 0xC025, "M-BP81A", "MX500 Optical Mouse", HAS_RES | HAS_SS | HAS_SSR },
    { VENDOR_LOGITECH, 0xC031, "M-UT58A", "iFeel Mouse (silver)", HAS_RES },
    { VENDOR_LOGITECH, 0xC501, "C-BA4-MSE", "Mouse Receiver", HAS_CSR },
    { VENDOR_LOGITECH, 0xC502, "C-UA3-DUAL", "Dual Receiver", HAS_CSR | USE_CH2},
    { VENDOR_LOGITECH, 0xC504, "C-BD9-DUAL", "Cordless Freedom Optical", HAS_CSR | USE_CH2 },
    { VENDOR_LOGITECH, 0xC505, "C-BG17-DUAL", "Cordless Elite Duo", HAS_SS | HAS_SSR | HAS_CSR | USE_CH2},
    { VENDOR_LOGITECH, 0xC506, "C-BF16-MSE", "MX700 Optical Mouse", HAS_SS | HAS_CSR },
    { VENDOR_LOGITECH, 0xC508, "C-BA4-MSE", "Cordless Optical TrackMan", HAS_SS | HAS_CSR },
    { VENDOR_LOGITECH, 0xC50B, "967300-0403", "Cordless MX Duo Receiver", HAS_SS|HAS_CSR },
    { VENDOR_LOGITECH, 0xC50E, "M-RAG97", "MX1000 Laser Mouse", HAS_SS | HAS_CSR },
    { VENDOR_LOGITECH, 0xC702, "C-UF15", "Receiver for Cordless Presenter", HAS_CSR },
    { 0, 0, 0, 0, 0 }
};

int main(int argc, char **argv)
{
    struct usb_bus *bus;
    struct usb_device *dev;
    int c;
    int resolution = 0;
    int cruise = 0;
    uint8_t status[8];
    int n;
    unsigned int yalv;
    int result;

    while (1) {
        int option_index = 0;
        static struct option long_options[] = {
            {"get-res", 0, 0, 'g'},
            {"get-cc", 0, 0, 'c'},
            {"help", 0, 0, 'h'},
            {"disable-cc", 0, 0, 'd'},
            {"enable-cc", 0, 0, 'e'},
            {"set-res", 1, 0, 's'},
            {"version", 0, 0, 'v'},
            {0, 0, 0, 0}
        };

        c = getopt_long  (argc, argv, "cdeghs:v",
                 long_options, &option_index);
        if (c == -1)
            break;

        switch (c) {
        case 'c':
	    cruise = 0;
            break;
        case 'd':
	    cruise = 1;
            break;
        case 'e':
	    cruise = 2;
            break;
        case 'g':
	    resolution = 0;
            break;
        case 'h':
	    usage();
	    return 0;
            break;
        case 's':
	    if (!strcmp("400", optarg))
		resolution = 400;
	    else if (!strcmp("800", optarg))
		resolution = 800;
	    else
		printf("Bad argument (should be 400 or 800)\n");
            break;
        case 'v':
	    version();
	    return 0;
            break;

        default:
	    usage();
	    break;
        }
    }

    usb_init();

    usb_find_busses();
    usb_find_devices();

    for (bus = usb_busses; bus; bus = bus->next) {
	for (dev = bus->devices; dev; dev = dev->next) {
	    for (n = 0; device_table[n].idVendor; n++)
		if ( (device_table[n].idVendor == dev->descriptor.idVendor) &&
		     (device_table[n].idProduct == dev->descriptor.idProduct) ) {
		    printf("%s/%s     %04X/%04X  \t%s \t%s\n",
			   bus->dirname,
			   dev->filename,
			   dev->descriptor.idVendor,
			   dev->descriptor.idProduct,
			   device_table[n].Model,
			   device_table[n].Name);

		    /* resolution switching */
		    if (0 != (device_table[n].flags & HAS_RES)) {
			if (0 == resolution) { /* user didn't specify or asked explicitly */
			    resolution = get_resolution(dev);
			    printf("   Resolution: ");
			    switch (resolution)
				{
				case 0: printf(" Not available (You probably aren't root)\n");
				    break;
				case 3: printf("400cpi\n");
				    break;
				case 4: printf("800cpi\n");
				    break;
				default: printf("(Unexpected result:%i)\n", resolution);
				    break;
				}
			} else {
			    if (400 == resolution)
				set_resolution(dev, 0x03);
			    else 
				set_resolution(dev, 0x04);
			}
		    }

		    /* smart scroll */
		    if (0 != (device_table[n].flags & HAS_SSR)) {
			if (0 == cruise) {
			    cruise = get_cruise(dev, device_table[n].flags & USE_CH2 );
			    printf("   Cruise Control / Smart Scroll: ");
			    switch (cruise)
				{
				case 0: printf("Disabled\n");
				    break;
				case 1: printf("Enabled\n");
				    break;
				default: printf("Unknown (Unexpected result)\n");
				    break;
				}
			} 
		    }
		    if (0 != (device_table[n].flags & HAS_SS)) {
		        if (0 == cruise) {
			    /* do nothing - we dealt with this case above */
			} else if (1 == cruise) {
			    disable_cruise(dev);
			} else if (2 == cruise) {
			    enable_cruise(dev);
			} else {
			    printf("Unexpected cruise value : %i\n", cruise);
			}
		    }

		    /* Cordless Status Reporting */
		    if (0 != (device_table[n].flags & HAS_CSR)) {
			for (yalv=0; yalv<sizeof(status); yalv++) {
			    status[yalv] = 0;
			}
			result = get_csr(dev, device_table[n].flags & USE_CH2, status);
			
			// printf("   Result: %x\n", result);
			//printf("   P0  = %x\n", status[1]);
			//printf("   P6  = %x\n", status[0]);
			//printf("   P4  = %x\n", status[2]);
			//printf("   P5  = %x ", status[3]);
			if (status[3] & 0x08) {
			    printf("   Channel 2    ");
			} else {
			    printf("   Channel 1    ");
			}
			printf("Battery: %i    ", status[3]&0x07);
			//printf("   P8  = %x\n", status[4]);
			//printf("   P9  = %x\n", status[5]);
			//printf("   PB0 = %x\n", status[6]);
			//printf("   PB1 = %x ", status[7]);
			if (status[7] & 0x40) {
			    printf("Two channel   ");
			} else {
			    printf("Single channel   ");
			}
			if (status[7] & 0x20) {
			    printf("800cpi support   ");
			} else {
			    printf("No 800cpi support   ");
			}
			if (status[7] & 0x10) {
			    printf("Horiz Roller   ");
			} else {
			    printf("No Horiz Roller   ");
			}
			if (status[7] & 0x08) {
			    printf("Vert Roller   ");
			} else {
			    printf("No Vert Roller   ");
			}
			if (0x7 == (status[7] & 0x7)) {
			    printf("More than 8 butt.\n");
			} else {
			    printf("%i butt.\n", ((status[7]&0x07)+2));
			}
		    }
		}
	}
    }
    return 0;

}

