/**
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010. 
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.  
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 * See the License for the specific language governing permissions and 
 * limitations under the License.
 *
 *
 *  Authors:
 *  2009-
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     Mischa Sall\'e <msalle@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl> 
 *
 *  2007-2009
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 *  2003-2007
 *     Martijn Steenbakkers <martijn@nikhef.nl>
 *     Gerben Venekamp <venekamp@nikhef.nl>
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 */


/*!
    \file   lcmaps_vo_data.c
    \brief  LCMAPS utilities for creating and accessing VO data structures
    \author Martijn Steenbakkers for the EU DataGrid.

    The interface is composed of:
    -# lcmaps_createVoData(): create a VoData structure
    -# lcmaps_deleteVoData(): delete a VoData structure
    -# lcmaps_copyVoData(): copy (the contents of) a VoData structure
    -# lcmaps_printVoData(): print the contents of a VoData structure
    -# lcmaps_stringVoData(): cast a VoData structure into a string
*/

#define _XOPEN_SOURCE	500

/******************************************************************************
                             Include header files
******************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <errno.h>

#include "lcmaps_vo_data.h"
#include "lcmaps_log.h"

/******************************************************************************
                                Definitions
******************************************************************************/
#define VO_DATA_WHITESPACE_CHARS " \t\n"

/*
 * VO mapping section
 */
/******************************************************************************
Function:   lcmaps_createVoData
Description:
    Create a VoData structure (store a VO, group, (subgroup,) role, capability
    combination). Allocate the memory. To be freed with lcmaps_deleteVoData().

Parameters:
    vo: name of the VO
    group: name of the group
    subgroup: name of the subgroup (ignored for the moment)
    role: the role
    capability: the capability (whatever it is)
Returns:
    pointer to the VoData structure or NULL
******************************************************************************/
/*!
    \fn lcmaps_createVoData(
        const char * vo,
        const char * group,
        const char * subgroup,
        const char * role,
        const char * capability
    )
    \brief Create a VoData structure

    Create a VoData structure (store a VO, group, (subgroup,) role, capability
    combination). Allocate the memory. To be freed with lcmaps_deleteVoData().

    \param vo name of the VO
    \param group name of the group
    \param subgroup name of the subgroup (ignored for the moment)
    \param role the role
    \param capability the capability (whatever it is)

    \return pointer to the VoData structure or NULL
*/
lcmaps_vo_data_t *
lcmaps_createVoData(
    const char * vo,
    const char * group,
    const char * subgroup,
    const char * role,
    const char * capability
)
{
    lcmaps_vo_data_t * newVoData=NULL;

    newVoData = (lcmaps_vo_data_t *)malloc(sizeof(lcmaps_vo_data_t));
    if (!newVoData)
    {
        lcmaps_log(LOG_ERR,"lcmaps_createVoData(): error in malloc for new VoData structure\n");
        return NULL;
    }

    newVoData->vo = NULL;
    newVoData->group = NULL;
    newVoData->subgroup = NULL;
    newVoData->role = NULL;
    newVoData->capability = NULL;

    if (vo) {
	if ( (newVoData->vo = strdup(vo))==NULL)    {
	    lcmaps_log(LOG_ERR,"%s: Out of memory\n",__func__);
	    return NULL;
	}
    }
    if (group) {
	if ( (newVoData->group = strdup(group))==NULL)	{
	    lcmaps_log(LOG_ERR,"%s: Out of memory\n",__func__);
	    return NULL;
	}
    }
    if (subgroup) {
	if ( (newVoData->subgroup = strdup(subgroup))==NULL)	{
	    lcmaps_log(LOG_ERR,"%s: Out of memory\n",__func__);
	    return NULL;
	}
    }
    if (role) {
	if ( (newVoData->role = strdup(role))==NULL)	{
	    lcmaps_log(LOG_ERR,"%s: Out of memory\n",__func__);
	    return NULL;
	}
    }
    if (capability) {
	if ( (newVoData->capability = strdup(capability))==NULL)	{
	    lcmaps_log(LOG_ERR,"%s: Out of memory\n",__func__);
	    return NULL;
	}
    }

    return newVoData;
}

/******************************************************************************
Function:   lcmaps_deleteVoData
Description:
    Delete a VoData structure that was previously created with lcmaps_createVoData().
    The pointer to the VoData structure is finally set to NULL;

Parameters:
    vo_data: pointer to a pointer to a VoData structure

Returns:
    0: success
    -1: failure (could not delete the structure or no structure found)
******************************************************************************/
/*!
    \fn lcmaps_deleteVoData(
        lcmaps_vo_data_t ** vo_data
    )
    \brief Delete a VoData structure

    Delete a VoData structure that was previously created with lcmaps_createVoData().
    The pointer to the VoData structure is finally set to NULL;

    \param vo_data pointer to a pointer to a VoData structure

    \retval 0 in case of success
    \retval -1 in case of failure
*/
int
lcmaps_deleteVoData(
    lcmaps_vo_data_t ** vo_data
)
{
    if (!vo_data) {
        lcmaps_log(LOG_ERR, "lcmaps_deleteVoData(): empty pointer as input !\n");
        return -1;
    }

    if ( (*vo_data) )
    {
        if ( (*vo_data)->vo) free( (*vo_data)->vo );
        if ( (*vo_data)->group) free( (*vo_data)->group );
        if ( (*vo_data)->subgroup) free( (*vo_data)->subgroup );
        if ( (*vo_data)->role) free( (*vo_data)->role );
        if ( (*vo_data)->capability) free( (*vo_data)->capability );
        free( (*vo_data) );
    }
    else
    {
        lcmaps_log_debug(2,"lcmaps_deleteVoData(): no lcmaps_vo_data_t found\n");
    }
    *vo_data=NULL;
    return 0;
}

/******************************************************************************
Function:   lcmaps_cleanVoData
Description:
    Clean a VoData structure that was previously filled with lcmaps_copyVoData().
    The contents are freed and set to zero.

Parameters:
    vo_data: a pointer to a VoData structure

Returns:
    0: success
    -1: failure (could not clean the structure or no structure found)
******************************************************************************/
/*!
    \fn lcmaps_cleanVoData(
        lcmaps_vo_data_t * vo_data
    )
    \brief Clean a VoData structure

    Clean a VoData structure that was previously filled with lcmaps_copyVoData().
    The contents are freed and set to zero.

    \param vo_data a pointer to a VoData structure

    \retval 0 in case of success
    \retval -1 in case of failure
*/
int
lcmaps_cleanVoData(
    lcmaps_vo_data_t * vo_data
)
{
    if (!vo_data) {
        lcmaps_log(LOG_ERR, "lcmaps_cleanVoData():: no lcmaps_vo_data_t found\n");
        return -1;
    }
    else
    {
        if ( (vo_data)->vo)
        {
            free( (vo_data)->vo );
            vo_data->vo = NULL;
        }
        if ( (vo_data)->group) 
        {
            free( (vo_data)->group );
            vo_data->group = NULL;
        }
        if ( (vo_data)->subgroup) 
        {
            free( (vo_data)->subgroup );
            vo_data->subgroup = NULL;
        }
        if ( (vo_data)->role) 
        {
            free( (vo_data)->role );
            vo_data->role = NULL;
        }
        if ( (vo_data)->capability) 
        {
            free( (vo_data)->capability );
            vo_data->capability = NULL;
        }
    }
    return 0;
}

/******************************************************************************
Function:   lcmaps_copyVoData
Description:
    Copy a VoData structure into an empty VoData structure which has to exist.

Parameters:
    dst_vo_data pointer to a empty VoData structure that should be filled
    src_vo_data pointer to the VoData structure that should be copied

Returns:
    0: success
    -1: failure (either src_vo_data or dst_vo_data was empty)
******************************************************************************/
/*!
    \fn lcmaps_copyVoData(
        lcmaps_vo_data_t * dst_vo_data,
        const lcmaps_vo_data_t * src_vo_data
    )
    \brief Copy a VoData structure into an empty VoData structure

    Copy a VoData structure into an empty VoData structure which has to exist.

    \param dst_vo_data pointer to a empty VoData structure that should be filled
    \param src_vo_data pointer to the VoData structure that should be copied

    \retval 0 success
    \retval -1 failure (either src_vo_data or dst_vo_data was empty)
*/
int
lcmaps_copyVoData(
    lcmaps_vo_data_t * dst_vo_data,
    const lcmaps_vo_data_t * src_vo_data
)
{
    if ( (dst_vo_data) && (src_vo_data) )
    {
        if (src_vo_data->vo)	{
            if ( (dst_vo_data->vo = strdup(src_vo_data->vo))==NULL)
		return -1;
	} else
            dst_vo_data->vo = NULL;
        if (src_vo_data->group)	{
            if ( (dst_vo_data->group = strdup(src_vo_data->group))==NULL)
		return -1;
        } else
            dst_vo_data->group = NULL;
        if (src_vo_data->subgroup)  {
            if ( (dst_vo_data->subgroup = strdup(src_vo_data->subgroup))==NULL)
		return -1;
        } else
            dst_vo_data->subgroup = NULL;
        if (src_vo_data->role)	{
            if ( (dst_vo_data->role = strdup(src_vo_data->role))==NULL)
		return -1;
        } else
            dst_vo_data->role = NULL;
        if (src_vo_data->capability)	{
            if ( (dst_vo_data->capability = strdup(src_vo_data->capability))==NULL)
		return -1;
        } else
            dst_vo_data->capability = NULL;

        return 0;
    }
    else
    {
        return -1;
    }
}

/******************************************************************************
Function:   lcmaps_printVoData
Description:
    Print the contents of a VoData structure

Parameters:
    debug_level: debug_level for which the contents will be printed
    vo_data: pointer to a VoData structure

Returns:
     0 (always)
******************************************************************************/
/*!
    \fn lcmaps_printVoData(
        int debug_level,
        const lcmaps_vo_data_t * vo_data
    )
    \brief Print the contents of a VoData structure

    \param vo_data     pointer to a VoData structure
    \param debug_level debug_level for which the contents will be printed

    \return 0 (always)
*/
int
lcmaps_printVoData(
    int debug_level,
    const lcmaps_vo_data_t * vo_data
)
{
    if (vo_data)
    {
        lcmaps_log_debug(debug_level,"lcmaps_printVoData(): address of vo data struct: %p\n", (const void*)vo_data);
        lcmaps_log_debug(debug_level,"lcmaps_printVoData():                        VO: %s\n", vo_data->vo);
        lcmaps_log_debug(debug_level,"lcmaps_printVoData():                     GROUP: %s\n", vo_data->group);
        lcmaps_log_debug(debug_level,"lcmaps_printVoData():                  SUBGROUP: %s\n", vo_data->subgroup);
        lcmaps_log_debug(debug_level,"lcmaps_printVoData():                      ROLE: %s\n", vo_data->role);
        lcmaps_log_debug(debug_level,"lcmaps_printVoData():                CAPABILITY: %s\n", vo_data->capability);
    }
    else
    {
        lcmaps_log_debug(debug_level,"lcmaps_printVoData(): empty pointer to vo data struct\n");
    }
    return 0;
}

/******************************************************************************
Function:   lcmaps_stringVoData
Description:
    Cast a VoData structure into a string

    The user of this function should create the buffer of size nchars beforehand.
    In buffer a string like the following will be written:
    "/VO=fred/GROUP=fred/flintstone/ROLE=director/CAPABILITY=destroy"

    Currently the SUBGROUP entry is ignored. Only if the information is present in the 
    VoData structure, it is added to the string.
    Both data for VO and GROUP are required (might change).


Parameters:
    vo_data: pointer to a VoData structure
    buffer: pointer to character array of size nchars
    nchars: size of character array

Returns:
    0: success
    -1: failure
******************************************************************************/
/*!
    \fn lcmaps_stringVoData(
        const lcmaps_vo_data_t * vo_data,
        char * buffer,
        int nchars
    )
    \brief Cast a VoData structure into a string

    The user of this function should create the buffer of size nchars beforehand.
    In buffer a string like the following will be written:
    "/VO=fred/GROUP=fred/flintstone/ROLE=director/CAPABILITY=destroy"

    Currently the SUBGROUP entry is ignored. Only if the information is present in the 
    VoData structure, it is added to the string.
    Both data for VO and GROUP are required (might change).

    \param vo_data pointer to a VoData structure
    \param buffer pointer to character array of size nchars
    \param nchars size of character array

    \retval 0 in case of success
    \retval -1 in case of failure
*/
int
lcmaps_stringVoData(
    const lcmaps_vo_data_t * vo_data,
    char * buffer,
    int nchars
)
{
    int totalchars;
    char * strptr=NULL;
    char * bufptr=NULL;
    int    buflen=0;

    bufptr=buffer;
    buflen=nchars;

    /* write "VO=" */
    if ( vo_data->vo == NULL )
    {
        lcmaps_log(LOG_ERR,"lcmaps_stringVoData(): error no VO found\n");
        return -1;
    }
    /* Skip over leading whitespace */
    strptr=vo_data->vo;
    strptr += strspn(strptr, VO_DATA_WHITESPACE_CHARS);
    if ( (*strptr!='\0') && (strncmp(strptr,"NULL",(size_t)4)) )
    {
        /* Convention for solaris 2.8 and glibc-2.1 and higher:
         * totalchars is the number of chars written (excluding \0) if enough space had been 
         * available.
         * negative: error
         */
        totalchars=snprintf(bufptr,(size_t)buflen,"/VO=%s",strptr);
        if ( totalchars < 0 )
        {
            lcmaps_log(LOG_ERR,"lcmaps_stringVoData(): error in snprintf(): %s\n",
		    strerror(errno));
            return -1;
        }
	else if ( (totalchars+1) > buflen )
        {
            lcmaps_log(LOG_ERR,"lcmaps_stringVoData(): could not write all characters into buffer for VO. Excess of characters: %d\n",totalchars+1-buflen);
            return -1;
        }
        else
        {
            bufptr+=totalchars;
            buflen-=totalchars;
        }
    }
    else
    {
        lcmaps_log(LOG_ERR,"lcmaps_stringVoData(): error no VO found\n");
        return -1;
    }

    /* write "GROUP=" */
    if ( vo_data->group == NULL )
    {
        lcmaps_log(LOG_ERR,"lcmaps_stringVoData(): error no VO-group found\n");
        return -1;
    }
    /* Skip over leading whitespace */
    strptr=vo_data->group;
    strptr += strspn(strptr, VO_DATA_WHITESPACE_CHARS);
    if ( (*strptr!='\0') && (strncmp(strptr,"NULL",(size_t)4)) )
    {
        totalchars=snprintf(bufptr,(size_t)buflen,"/GROUP=%s",strptr);
        if ( totalchars < 0 )
        {
            lcmaps_log(LOG_ERR,"lcmaps_stringVoData(): error in snprintf(): %s\n",
		    strerror(errno));
            return -1;
        }
	else if ( (totalchars+1) > buflen )
        {
            lcmaps_log(LOG_ERR,"lcmaps_stringVoData(): could not write all characters into buffer for GROUP. Excess of characters: %d\n",totalchars+1-buflen);
            return -1;
        }
        else
        {
            bufptr+=totalchars;
            buflen-=totalchars;
        }
    }
    else
    {
        lcmaps_log(LOG_ERR,"lcmaps_stringVoData(): error no VO-group found\n");
        return -1;
    }

    /* Skip subgroup for the moment (not clear how VOMS will evolve in this respect) */

    /* write "ROLE=" */
    if ( vo_data->role != NULL )
    {
        /* Skip over leading whitespace */
        strptr=vo_data->role;
        strptr += strspn(strptr, VO_DATA_WHITESPACE_CHARS);
        if ( (*strptr!='\0') && (strncmp(strptr,"NULL",(size_t)4)) )
        {
            totalchars=snprintf(bufptr,(size_t)buflen,"/ROLE=%s",strptr);
            if ( totalchars < 0 )
            {
                lcmaps_log(LOG_ERR,"lcmaps_stringVoData(): error in snprintf(): %s\n",
			strerror(errno));
                return -1;
            }
	    else if ( (totalchars+1) > buflen )
            {
                lcmaps_log(LOG_ERR,"lcmaps_stringVoData(): could not write all characters into buffer for ROLE. Excess of characters: %d\n",totalchars+1-buflen);
                return -1;
            }
            else
            {
                bufptr+=totalchars;
                buflen-=totalchars;
            }
        }
    }

    /* write "CAPABILITY=" */
    if ( vo_data->capability != NULL )
    {
        /* Skip over leading whitespace */
        strptr=vo_data->capability;
        strptr += strspn(strptr, VO_DATA_WHITESPACE_CHARS);
        if ( (*strptr!='\0') && (strncmp(strptr,"NULL",(size_t)4)) )
        {
            totalchars=snprintf(bufptr,(size_t)buflen,"/CAPABILITY=%s",strptr);
            if ( totalchars < 0 )
            {
                lcmaps_log(LOG_ERR,"lcmaps_stringVoData(): error in snprintf(): %s\n",
			strerror(errno));
                return -1;
            }
            else if ( (totalchars+1) > buflen )
            {
                lcmaps_log(LOG_ERR,"lcmaps_stringVoData(): could not write all characters into buffer for CAPABILITY. Excess of characters: %d\n",totalchars+1-buflen);
                return -1;
            }
            else
            {
                bufptr+=totalchars;
                buflen-=totalchars;
            }
        }
    }
    return 0;
}

/*
 * VO mapping section
 */
/******************************************************************************
Function:   lcmaps_createVoMapping
Description:
    Create a VoMapping structure store the VO information string (or FQAN) in combination
    with the corresponding gid and groupname.
    Allocate the memory. To be freed with lcmaps_deleteVoMapping().

Parameters:
    vo_data_string: the VO information string (or FQAN)
    groupname:      the (POSIX) group name
    gid:            the (POSIX) gid
Returns:
    pointer to the VoMapping structure or NULL
******************************************************************************/
/*!
    \fn lcmaps_createVoMapping(
        const char * vo_data_string,
        const char * groupname,
        const gid_t  gid
    )
    \brief Create a VoMapping structure

    Create a VoMapping structure store the VO information string (or FQAN) in combination
    with the corresponding gid.
    Allocate the memory. To be freed with lcmaps_deleteVoMapping().

    \param vo_data_string the VO information string (or FQAN)
    \param groupname the (POSIX) group name
    \param gid the (POSIX) GID

    \return pointer to the VoMapping structure or NULL
*/
lcmaps_vo_mapping_t *
lcmaps_createVoMapping(
    const char * vo_data_string,
    const char * groupname,
    const gid_t  gid
)
{
    lcmaps_vo_mapping_t * newVoMapping=NULL;

    newVoMapping = (lcmaps_vo_mapping_t *)malloc(sizeof(lcmaps_vo_mapping_t));
    if (!newVoMapping)
    {
        lcmaps_log(LOG_ERR,"lcmaps_createVoMapping(): error in malloc for new VoMapping structure\n");
        return NULL;
    }

    newVoMapping->vostring  = NULL;
    newVoMapping->groupname = NULL;
    newVoMapping->gid       = LCMAPS_NO_GID;

    if (vo_data_string)	{
	if ( (newVoMapping->vostring = strdup(vo_data_string))==NULL )	{
	    lcmaps_log(LOG_ERR, "%s: Out of memory\n", __func__);
	    return NULL;
	}
    }
	    
    if (vo_data_string)	{
	if ( (newVoMapping->groupname = strdup(groupname))==NULL )	{
	    lcmaps_log(LOG_ERR, "%s: Out of memory\n", __func__);
	    return NULL;
	}
    }

    if (gid) newVoMapping->gid = gid;

    return newVoMapping;
}

/******************************************************************************
Function:   lcmaps_deleteVoMapping
Description:
    Delete a VoMapping structure that was previously created with lcmaps_createVoMapping().
    The pointer to the VoMapping structure is finally set to NULL;

Parameters:
    vo_mapping: pointer to a pointer to a VoMapping structure

Returns:
    0: success
    -1: failure (could not delete the structure or no structure found)
******************************************************************************/
/*!
    \fn lcmaps_deleteVoMapping(
        lcmaps_vo_mapping_t ** vo_mapping
    )
    \brief Delete a VoMapping structure

    Delete a VoMapping structure that was previously created with lcmaps_createVoMapping().
    The pointer to the VoMapping structure is finally set to NULL;

    \param vo_mapping pointer to a pointer to a VoMapping structure

    \retval 0 in case of success
    \retval -1 in case of failure
*/
int
lcmaps_deleteVoMapping(
    lcmaps_vo_mapping_t ** vo_mapping
)
{
    if (!vo_mapping) {
        lcmaps_log(LOG_ERR, "lcmaps_deleteVoMapping(): empty pointer as input !\n");
        return -1;
    }

    if ( (*vo_mapping) )
    {
        if ( (*vo_mapping)->vostring) free( (*vo_mapping)->vostring );
        if ( (*vo_mapping)->groupname) free( (*vo_mapping)->groupname );
        free( (*vo_mapping) );
    }
    else
    {
        lcmaps_log_debug(2,"lcmaps_deleteVoMapping(): no lcmaps_vo_mapping_t found\n");
    }
    *vo_mapping=NULL;
    return 0;
}

/******************************************************************************
Function:   lcmaps_cleanVoMapping
Description:
    Clean a VoMapping structure that was previously filled with lcmaps_copyVoMapping().
    The contents are freed and set to zero.

Parameters:
    vo_mapping: a pointer to a VoMapping structure

Returns:
    0: success
    -1: failure (could not clean the structure or no structure found)
******************************************************************************/
/*!
    \fn lcmaps_cleanVoMapping(
        lcmaps_vo_mapping_t * vo_mapping
    )
    \brief Clean a VoMapping structure

    Clean a VoMapping structure that was previously filled with lcmaps_copyVoMapping().
    The contents are freed and set to zero.

    \param vo_mapping a pointer to a VoMapping structure

    \retval 0 in case of success
    \retval -1 in case of failure
*/
int
lcmaps_cleanVoMapping(
    lcmaps_vo_mapping_t * vo_mapping
)
{
    if (!vo_mapping) {
        lcmaps_log(0, "lcmaps_cleanVoMapping():: no lcmaps_vo_mapping_t found\n");
        return -1;
    }
    else
    {
        if ( (vo_mapping)->vostring)
        {
            free( (vo_mapping)->vostring );
            vo_mapping->vostring = NULL;
        }

        if ( (vo_mapping)->groupname)
        {
            free( (vo_mapping)->groupname );
            vo_mapping->groupname = NULL;
        }
    }
    return 0;
}

/******************************************************************************
Function:   lcmaps_copyVoMapping
Description:
    Copy a VoMapping structure into an empty VoMapping structure which has to exist.

Parameters:
    dst_vo_mapping pointer to a empty VoMapping structure that should be filled
    src_vo_mapping pointer to the VoMapping structure that should be copied

Returns:
    0: success
    -1: failure (either src_vo_mapping or dst_vo_mapping was empty)
******************************************************************************/
/*!
    \fn lcmaps_copyVoMapping(
        lcmaps_vo_mapping_t * dst_vo_mapping,
        const lcmaps_vo_mapping_t * src_vo_mapping
    )
    \brief Copy a VoMapping structure into an empty VoMapping structure

    Copy a VoMapping structure into an empty VoMapping structure which has to exist.

    \param dst_vo_mapping pointer to a empty VoMapping structure that should be filled
    \param src_vo_mapping pointer to the VoMapping structure that should be copied

    \retval 0 success
    \retval -1 failure (either src_vo_mapping or dst_vo_mapping was empty)
*/
int
lcmaps_copyVoMapping(
    lcmaps_vo_mapping_t * dst_vo_mapping,
    const lcmaps_vo_mapping_t * src_vo_mapping
)
{
    if ( (dst_vo_mapping) && (src_vo_mapping) )
    {
        dst_vo_mapping->gid = src_vo_mapping->gid;
        if (src_vo_mapping->vostring)	{
	    if ( (dst_vo_mapping->vostring = strdup(src_vo_mapping->vostring)) == NULL)
		return -1;
	} else
            dst_vo_mapping->vostring = NULL;

        if (src_vo_mapping->groupname)	{
            if ( (dst_vo_mapping->groupname = strdup(src_vo_mapping->groupname))==NULL)
		return -1;
	} else
            dst_vo_mapping->groupname = NULL;

        return 0;
    }
    else
    {
        return -1;
    }
}

/******************************************************************************
Function:   lcmaps_printVoMapping
Description:
    Print the contents of a VoMapping structure

Parameters:
    debug_level: debug_level for which the contents will be printed
    vo_mapping:  pointer to a VoMapping structure

Returns:
     0 (always)
******************************************************************************/
/*!
    \fn lcmaps_printVoMapping(
        int debug_level,
        const lcmaps_vo_mapping_t * vo_mapping
    )
    \brief Print the contents of a VoMapping structure

    \param vo_mapping  pointer to a VoMapping structure
    \param debug_level debug_level for which the contents will be printed

    \return 0 (always)
*/
int
lcmaps_printVoMapping(
    int debug_level,
    const lcmaps_vo_mapping_t * vo_mapping
)
{
    if (vo_mapping)
    {
        if (vo_mapping->groupname)
            lcmaps_log(debug_level,"lcmaps_printVoMapping(): VO string: \"%s\" -> mapped unix-group: %d(%s)\n", vo_mapping->vostring, (int)vo_mapping->gid, vo_mapping->groupname);
        else    
            lcmaps_log(debug_level,"lcmaps_printVoMapping(): VO string: \"%s\" -> mapped unix-group: %d\n", vo_mapping->vostring, (int)vo_mapping->gid);
    }
    else
    {
        lcmaps_log_debug(debug_level,"lcmaps_printVoMapping(): empty pointer to vo mapping struct\n");
    }
    return 0;
}

/******************************************************************************
CVS Information:
    $Source: /srv/home/dennisvd/svn/mw-security/lcmaps/src/pluginmanager/lcmaps_vo_data.c,v $
    $Date: 2014-07-08 10:39:48 +0200 (Tue, 08 Jul 2014) $
    $Revision: 17848 $
    $Author: msalle $
******************************************************************************/
