# Copyright (C) 2022 Canonical Ltd.
# Author: Benjamin Drung <benjamin.drung@canonical.com>
#
# This program is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by the
# Free Software Foundation; either version 2 of the License, or (at your
# option) any later version.  See http://www.gnu.org/copyleft/gpl.html for
# the full text of the license.

"""Add image build information if available."""

import os

_BUILD_INFO_KEY_MAPPING = {"build_name": "CloudBuildName", "serial": "CloudSerial"}


def _add_cloud_build_info(report):
    """Add cloud build information if available.

    /etc/cloud/build.info is generated by livecd-rootfs.
    """
    if not os.path.isfile("/etc/cloud/build.info"):
        return
    report.add_tags(["cloud-image"])
    with open("/etc/cloud/build.info", encoding="utf-8") as build_info:
        for line in build_info:
            try:
                key, value = line.split(":", 1)
            except ValueError:
                continue
            if key not in _BUILD_INFO_KEY_MAPPING:
                continue
            report[_BUILD_INFO_KEY_MAPPING[key]] = value.strip()


def add_info(report, unused_ui):
    """Add image build information if available."""
    _add_cloud_build_info(report)
