use ui;
use layout;
use progvis:net;
use progvis:program;
use core:geometry;
use core:io;

/**
 * Behavior for revisiting an already solved problem.
 */
class RevisitBehavior extends DefaultBehavior {
	private Str title;

	init(MainWin window, Str title) {
		init(window) { title = title; }
	}

	Str? allowSpawnThread() : override { "Can not spawn new threads this way when solving problems."; }
	Str? allowReload() : override { "Can not reload the current problem. Use Run -> Restart instead, or open a new problem."; }
	Str? allowModelCheck() : override { "The model checker is disabled while looking at other solutions."; }
	Bool allowTrackMemory() : override { false; }

	ProblemPanel? panel() : override {
		ProblemPanel("Find an error in \"${title}\" (can not submit a solution)");
	}
}


/**
 * Behavior asking the student to find and correct an error in the implementation.
 */
class DebugProblemBehavior extends Behavior {
	private MainWin window;
	private Client client;
	private Problem problem;
	private Url testCode;
	private Url implCode;
	private StrBuf operations;
	private Bool done;

	init(MainWin window, Client client, problems:Action action) {
		init() {
			window = window;
			client = client;
			problem = action.problem;
			testCode = action.testCode;
			implCode = action.implCode;
		}
	}

	ProblemPanel? panel() : override {
		ProblemPanel p("Find an error in \"${problem.title}\"!");
		p.button("Restart", &this.onRestartClicked());
		p.button("Open test", &this.onOpenMain());
		p.button("Open implementation", &this.onOpenImpl());
		p;
	}

	private void onOpenImpl() {
		(spawn window.openEditor(implCode)).detach();
	}

	private void onOpenMain() {
		(spawn window.openEditor(testCode)).detach();
	}

	private void onRestartClicked() {
		(spawn window.onRestart()).detach();
	}

	void onProgramError(ProgramError error) : override {
		if (done)
			return;
		done = true;

		(spawn submitError(error)).detach();
	}

	private void submitError(ProgramError error) {
		(spawn submitAttempt(operations.toS, error.type)).detach();

		window.updateBehavior(ImproveBehavior(window, client, problem, testCode, implCode, error.type));
	}

	void onUserAction(Str action) : override {
		operations << action << ";";
	}

	void onRestart() : override {
		// Save the solution!
		Str ops = operations.toS();
		if (ops.any) {
			(spawn submitAttempt(ops, null)).detach();
		}

		operations = StrBuf();
	}

	private void submitAttempt(Str operations, Str? error) {
		client.query(NewAttemptRequest(problem.impl.id, problem.test.id, operations, error));
	}

	Str? allowSpawnThread() : override { "Can not spawn new threads this way when solving problems."; }
	Str? allowModelCheck() : override { "Try to find the error yourself using the step buttons for each thread."; }
	Str? allowReload() : override { "Can not reload the current problem. Use Run -> Restart instead, or open a new problem."; }
	Bool allowTrackMemory() : override { false; }
}


/**
 * Shared base for ImproveBehavior and TestBehavior.
 */
class UploadableBehavior extends DefaultBehavior {
	protected Client client;
	protected Problem problem;
	protected Url testCode;
	protected Url implCode;
	protected Bool submitted;

	init(MainWin window, Client client, Problem problem, Url testCode, Url implCode) {
		init(window) {
			client = client;
			problem = problem;
			testCode = testCode;
			implCode = implCode;
			submitted = false;
		}
	}

	// We encourage reloads!
	Str? allowReload() : override { null; }

	// Don't allow spawning threads arbitrarily.
	Str? allowSpawnThread() : override { "Can not spawn new threads this way when solving problems."; }

	// Don't allow model checking, that would be too easy.
	Str? allowModelCheck() : override { "The model checker is disabled for online problems."; }

	// Don't change 'track memory'.
	Bool allowTrackMemory() : override { false; }

	// Add a panel that explains it all.
	ProblemPanel? panel() : override {
		ProblemPanel p(panelText());
		p.button("Open test", &this.onOpenMain());
		p.button("Open impl.", &this.onOpenImpl());
		p.button("Reload", &this.onReload());
		p.button("Submit", &this.onSubmit());
		p;
	}

	// Title for the panel.
	protected Str panelText() : abstract;

	private void onSubmit() {
		// Avoid double submissions.
		if (submitted)
			return;
		submitted = true;

		(spawn doSubmitHelp()).detach();
	}

	private void doSubmitHelp() {
		// Possibly re-set 'submitted' if the user cancelled the dialog.
		submitted = doSubmit();
	}

	// Returns true if submission suceeded.
	protected Bool doSubmit() : abstract;

	private void onReload() {
		// Needed due to a bug in App on Windows.
		(spawn window.onReload()).detach();
	}

	private void onOpenImpl() {
		(spawn doOpenImpl()).detach();
	}

	protected void doOpenImpl() {
		window.openEditor(implCode);
	}

	private void onOpenMain() {
		(spawn doOpenMain()).detach();
	}

	protected void doOpenMain() {
		window.openEditor(testCode);
	}
}


/**
 * Behavior that lets the user modify the problem and submit an improved version.
 */
class ImproveBehavior extends UploadableBehavior {
	init(MainWin window, Client client, Problem problem, Url testCode, Url implCode, Str errorType) {
		init(window, client, problem, testCode, implCode) {}

		StrBuf msg;
		msg << "You found a problem of type: " << errorType << "\n\n";
		msg << "The next step is to fix the problem you found:\n";
		msg << "1. Select \"Open impl.\" if you have not done so already.\n";
		msg << "2. Modify the code and save the file in your editor.\n";
		msg << "3. Select \"Reload\" to load your changes and see if they work.\n";
		msg << "4. Repeat steps 2-3 as necessary.\n";
		msg << "5. Click \"Submit\" in the green box when you are done.\n";
		msg << "Note: You can only modify the implementation, any chainges to the test file will be ignored.\n";
		showMessage(window, "Improve the code", msg.toS);
	}

	protected Str panelText() : override {
		"Try to fix the problem you found!";
	}

	protected Bool doSubmit() : override {
		Code implCode(window.currentSource(), problem.impl.language);
		progvis:problems:SolutionUploadDlg dlg(client, problem, null, implCode);
		if (dlg.show(window) >= 0) {
			window.updateBehavior(DefaultBehavior(window));
			return true;
		}
		return false;
	}

	protected void doOpenMain() {
		showMessage(window, "Information", "Note: You can not modify the test program in this problem. Any changes you make will be ignored.");
		super:doOpenMain();
	}
}


/**
 * Behavior similar to "ImproveBehavior" above, but that modifies the main file.
 */
class TestBehavior extends UploadableBehavior {
	init(MainWin window, Client client, Problem problem, Url testCode, Url implCode) {
		init(window, client, problem, testCode, implCode) {}

		(spawn infoMessage()).detach();
	}

	private void infoMessage() {
		sleep(200 ms);
		StrBuf msg;
		msg << "This implementation currently works as expected for the current test\n";
		msg << "program, but that does not mean the implementation is correct. Modify\n";
		msg << "the test program so that it finds a bug in the implementation:\n";
		msg << "1. Select \"Open test\" to show the test program.\n";
		msg << "2. Edit and save the file in your editor.\n";
		msg << "3. Select \"Reload\" to load your changes and see what happens.\n";
		msg << "4. Repeat steps 2-3 as necessary.\n";
		msg << "5. Click \"Submit\" in the green box when you are done.\n";
		showMessage(window, "Improve the test program", msg.toS);
	}

	protected Str panelText() : override {
		"Improve the test program to find a bug!";
	}

	protected Bool doSubmit() : override {
		Code testCode(window.currentSource(), problem.test.language);
		progvis:problems:SolutionUploadDlg dlg(client, problem, testCode, null);
		if (dlg.show(window) >= 0) {
			window.updateBehavior(DefaultBehavior(window));
			return true;
		}
		return false;
	}

	protected void doOpenImpl() : override {
		showMessage(window, "Information", "Note: You can not modify the implementation in this problem. Any changes you make will be ignored.");
		super:doOpenImpl();
	}
}
