use core:io;
use util:serialize;

/**
 * HTTP/HTTPS protocol.
 */
class HttpProtocol : extends core:io:Protocol, serializable {
	// Are we https?
	private Bool isSecure;

	// Create.
	init(Bool secure) {
		init { isSecure = secure; }
	}

	// Check if we are https.
	Bool secure() : inline { isSecure; }

	// Compare parts.
	Bool partEq(Str a, Str b) : override { a == b; }

	// Hash parts.
	Nat partHash(Str a) : override { a.hash(); }

	// Output.
	void toS(StrBuf to) : override {
		if (secure)
			to << "https://";
		else
			to << "http://";
	}

	// Compare.
	protected Bool isEqualTo(Protocol other) : override {
		if (!super:isEqualTo(other))
			return false;

		unless (other as HttpProtocol)
			return false;

		return secure == other.secure;
	}

	// Format an url.
	Str format(Url url) {
		StrBuf out;

		if (secure)
			out << "https:/";
		else
			out << "http:/";

		for (i, part in url) {
			out << "/";
			if (i == 0)
				out << part; // The hostname part is special, we don't want to mangle : there!
			else
				out << escapeUrl(part);
		}
		if (url.dir)
			out << "/";

		if (url as QueryUrl) {
			if (url.parameters.any) {
				out << "?";
				Bool first = true;
				for (k, v in url.parameters) {
					if (!first)
						out << "&";
					first = false;
					out << escapeUrlParam(k) << "=" << escapeUrlParam(v);
				}
			}
		}

		out.toS;
	}

	// Read an URL.
	IStream read(Url url) {
		return Client().request(url);
	}
}

// Create HTTP urls conveniently.
QueryUrl httpUrl(Str host) {
	QueryUrl(HttpProtocol(false), [host]);
}

QueryUrl httpsUrl(Str host) {
	QueryUrl(HttpProtocol(true), [host]);
}
