#include <cppunit/extensions/HelperMacros.h>
#include "../ResourceEditor.h"

#include <stdlib.h>

extern unsigned char original_pe[8704];

class CResourceEditorTest : public CppUnit::TestFixture {

  CPPUNIT_TEST_SUITE( CResourceEditorTest );
  CPPUNIT_TEST( testCorrectness );
  CPPUNIT_TEST( testBMP );
  CPPUNIT_TEST_SUITE_END();

public:
  enum { defnameid = 1337, deflangid = 1033 };

  void testCorrectness() {
    CResourceEditor re(original_pe, sizeof(original_pe));

    DWORD size;

    // get size
    size = re.Save(NULL, size);
    unsigned char *saved_pe = new unsigned char[size];

    // save
    int rc = re.Save(saved_pe, size);
    CPPUNIT_ASSERT_EQUAL( rc, 0 );

    // compare
    CPPUNIT_ASSERT_EQUAL( (DWORD) sizeof(original_pe), size );
    CPPUNIT_ASSERT_EQUAL( 0, memcmp(saved_pe, original_pe, size) );

    delete [] saved_pe;
  }

  void testBMP() {
    CResourceEditor re(original_pe, sizeof(original_pe));
    static const BYTE file_12_4bpp [] = { // BMP with the old header
      66,77,134,0,0,0,0,0,0,0,74,0,0,0,12,0,0,0,19,0,5,0,1,0,4,0,222,24,0,74,74,74,0,255,0,255,255,255,
      255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,
      255,255,255,255,255,255,255,255,255,255,255,255,2,34,2,0,2,32,0,32,0,32,0,0,2,32,2,34,32,34,2,34,
      34,0,0,0,2,2,2,32,2,34,2,34,0,32,0,0,0,34,2,2,34,34,2,32,34,32,0,0,2,34,2,32,0,32,0,34,0,0,0,0
    };
    bool succ;
    int ressize, bmpfilehdrsize = 14;
    BYTE *resdata;

    CPPUNIT_ASSERT( (succ = re.UpdateResource(RT_BITMAP, defnameid, deflangid, (BYTE*) file_12_4bpp, sizeof(file_12_4bpp), CResourceEditor::TM_AUTO)) );
    if (succ)
    {
      ressize = re.GetResourceSize(RT_BITMAP, defnameid, deflangid);
      CPPUNIT_ASSERT( ressize == sizeof(file_12_4bpp) - bmpfilehdrsize );
      if ((resdata = re.GetResource(RT_BITMAP, defnameid, deflangid)))
      {
        CPPUNIT_ASSERT_EQUAL( 0, memcmp(resdata, file_12_4bpp + bmpfilehdrsize, ressize) );
        re.FreeResource(resdata);
      }
    }
  }

};

CPPUNIT_TEST_SUITE_REGISTRATION( CResourceEditorTest );

unsigned char original_pe[8704] = {  
  77, 90, 144, 0, 3, 0, 0, 0, 4, 0,
  0, 0, 255, 255, 0, 0, 184, 0, 0, 0, 0,
  0, 0, 0, 64, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 200, 0, 0, 0, 14,
  31, 186, 14, 0, 180, 9, 205, 33, 184, 1, 76,
  205, 33, 84, 104, 105, 115, 32, 112, 114, 111, 103,
  114, 97, 109, 32, 99, 97, 110, 110, 111, 116, 32,
  98, 101, 32, 114, 117, 110, 32, 105, 110, 32, 68,
  79, 83, 32, 109, 111, 100, 101, 46, 13, 13, 10,
  36, 0, 0, 0, 0, 0, 0, 0, 252, 249, 48,
  199, 184, 152, 94, 148, 184, 152, 94, 148, 184, 152,
  94, 148, 184, 152, 95, 148, 168, 152, 94, 148, 59,
  144, 3, 148, 189, 152, 94, 148, 236, 187, 111, 148,
  185, 152, 94, 148, 127, 158, 88, 148, 185, 152, 94,
  148, 82, 105, 99, 104, 184, 152, 94, 148, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 80, 69, 0, 0, 76, 1, 4, 0,
  86, 34, 23, 68, 0, 0, 0, 0, 0, 0, 0,
  0, 224, 0, 15, 1, 11, 1, 6, 0, 0, 4,
  0, 0, 0, 26, 0, 0, 0, 0, 0, 0, 126,
  18, 0, 0, 0, 16, 0, 0, 0, 32, 0, 0,
  0, 0, 64, 0, 0, 16, 0, 0, 0, 2, 0,
  0, 4, 0, 0, 0, 0, 0, 0, 0, 4, 0,
  0, 0, 0, 0, 0, 0, 0, 96, 0, 0, 0,
  4, 0, 0, 0, 0, 0, 0, 2, 0, 0, 0,
  0, 0, 16, 0, 0, 16, 0, 0, 0, 0, 16,
  0, 0, 16, 0, 0, 0, 0, 0, 0, 16, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 72,
  32, 0, 0, 60, 0, 0, 0, 0, 80, 0, 0,
  168, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  32, 0, 0, 72, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 46, 116,
  101, 120, 116, 0, 0, 0, 160, 2, 0, 0, 0,
  16, 0, 0, 0, 4, 0, 0, 0, 4, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 32, 0, 0, 96, 46, 114, 100, 97, 116, 97,
  0, 0, 240, 1, 0, 0, 0, 32, 0, 0, 0,
  2, 0, 0, 0, 8, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 64, 0, 0,
  64, 46, 100, 97, 116, 97, 0, 0, 0, 100, 18,
  0, 0, 0, 48, 0, 0, 0, 20, 0, 0, 0,
  10, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 64, 0, 0, 192, 46, 114, 115,
  114, 99, 0, 0, 0, 168, 3, 0, 0, 0, 80,
  0, 0, 0, 4, 0, 0, 0, 30, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  64, 0, 0, 64, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 139, 68, 36, 8, 129, 236, 12, 2, 0,
  0, 45, 16, 1, 0, 0, 83, 85, 86, 87, 15,
  132, 234, 1, 0, 0, 72, 15, 133, 79, 2, 0,
  0, 139, 132, 36, 40, 2, 0, 0, 102, 61, 2,
  0, 117, 28, 51, 219, 83, 255, 180, 36, 36, 2,
  0, 0, 255, 21, 60, 32, 64, 0, 83, 255, 21,
  56, 32, 64, 0, 233, 38, 2, 0, 0, 139, 200,
  193, 233, 16, 102, 131, 249, 1, 15, 133, 178, 0,
  0, 0, 51, 219, 190, 71, 1, 0, 0, 83, 83,
  189, 234, 3, 0, 0, 86, 85, 255, 180, 36, 48,
  2, 0, 0, 255, 21, 52, 32, 64, 0, 131, 248,
  255, 15, 132, 241, 1, 0, 0, 83, 83, 191, 233,
  3, 0, 0, 86, 87, 255, 180, 36, 48, 2, 0,
  0, 255, 21, 52, 32, 64, 0, 131, 248, 255, 15,
  132, 210, 1, 0, 0, 83, 83, 86, 87, 255, 180,
  36, 48, 2, 0, 0, 139, 61, 52, 32, 64, 0,
  255, 215, 15, 183, 4, 197, 104, 50, 64, 0, 83,
  83, 86, 85, 255, 180, 36, 48, 2, 0, 0, 193,
  224, 10, 137, 68, 36, 36, 255, 215, 15, 183, 4,
  197, 0, 48, 64, 0, 139, 76, 36, 16, 11, 200,
  141, 68, 36, 28, 81, 104, 84, 66, 64, 0, 80,
  255, 21, 48, 32, 64, 0, 131, 196, 12, 141, 68,
  36, 28, 80, 104, 235, 3, 0, 0, 255, 180, 36,
  40, 2, 0, 0, 255, 21, 44, 32, 64, 0, 233,
  101, 1, 0, 0, 102, 61, 1, 0, 15, 133, 91,
  1, 0, 0, 51, 219, 190, 71, 1, 0, 0, 83,
  83, 189, 234, 3, 0, 0, 86, 85, 255, 180, 36,
  48, 2, 0, 0, 255, 21, 52, 32, 64, 0, 131,
  248, 255, 15, 132, 53, 1, 0, 0, 83, 83, 191,
  233, 3, 0, 0, 86, 87, 255, 180, 36, 48, 2,
  0, 0, 255, 21, 52, 32, 64, 0, 131, 248, 255,
  15, 132, 22, 1, 0, 0, 106, 16, 106, 2, 255,
  21, 4, 32, 64, 0, 59, 195, 137, 68, 36, 16,
  15, 132, 0, 1, 0, 0, 80, 255, 21, 0, 32,
  64, 0, 83, 83, 86, 87, 255, 180, 36, 48, 2,
  0, 0, 139, 61, 52, 32, 64, 0, 137, 68, 36,
  44, 255, 215, 15, 183, 4, 197, 104, 50, 64, 0,
  83, 83, 86, 85, 255, 180, 36, 48, 2, 0, 0,
  193, 224, 10, 137, 68, 36, 40, 255, 215, 15, 183,
  4, 197, 0, 48, 64, 0, 139, 76, 36, 20, 11,
  200, 81, 104, 80, 66, 64, 0, 255, 116, 36, 32,
  255, 21, 48, 32, 64, 0, 131, 196, 12, 255, 116,
  36, 16, 255, 21, 16, 32, 64, 0, 255, 180, 36,
  32, 2, 0, 0, 255, 21, 40, 32, 64, 0, 133,
  192, 15, 132, 134, 0, 0, 0, 255, 21, 36, 32,
  64, 0, 255, 116, 36, 16, 106, 1, 255, 21, 32,
  32, 64, 0, 255, 21, 28, 32, 64, 0, 235, 108,
  51, 219, 83, 83, 104, 128, 0, 0, 0, 255, 180,
  36, 44, 2, 0, 0, 255, 21, 24, 32, 64, 0,
  139, 53, 52, 32, 64, 0, 51, 255, 189, 234, 3,
  0, 0, 255, 183, 4, 48, 64, 0, 83, 104, 67,
  1, 0, 0, 85, 255, 180, 36, 48, 2, 0, 0,
  255, 214, 131, 199, 8, 129, 255, 104, 2, 0, 0,
  114, 223, 51, 237, 191, 233, 3, 0, 0, 255, 181,
  108, 50, 64, 0, 83, 104, 67, 1, 0, 0, 87,
  255, 180, 36, 48, 2, 0, 0, 255, 214, 131, 197,
  8, 129, 253, 224, 2, 0, 0, 114, 223, 95, 94,
  93, 51, 192, 91, 129, 196, 12, 2, 0, 0, 194,
  16, 0, 106, 0, 104, 0, 16, 64, 0, 106, 0,
  106, 101, 106, 0, 255, 21, 12, 32, 64, 0, 80,
  255, 21, 64, 32, 64, 0, 106, 0, 255, 21, 8,
  32, 64, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 220, 32, 0, 0, 234, 32, 0, 0,
  248, 32, 0, 0, 6, 33, 0, 0, 204, 32, 0,
  0, 0, 0, 0, 0, 40, 33, 0, 0, 56, 33,
  0, 0, 74, 33, 0, 0, 94, 33, 0, 0, 112,
  33, 0, 0, 128, 33, 0, 0, 146, 33, 0, 0,
  158, 33, 0, 0, 180, 33, 0, 0, 198, 33, 0,
  0, 210, 33, 0, 0, 0, 0, 0, 0, 132, 32,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 26,
  33, 0, 0, 0, 32, 0, 0, 156, 32, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 228, 33, 0,
  0, 24, 32, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 220, 32, 0, 0, 234, 32, 0, 0,
  248, 32, 0, 0, 6, 33, 0, 0, 204, 32, 0,
  0, 0, 0, 0, 0, 40, 33, 0, 0, 56, 33,
  0, 0, 74, 33, 0, 0, 94, 33, 0, 0, 112,
  33, 0, 0, 128, 33, 0, 0, 146, 33, 0, 0,
  158, 33, 0, 0, 180, 33, 0, 0, 198, 33, 0,
  0, 210, 33, 0, 0, 0, 0, 0, 0, 0, 2,
  71, 108, 111, 98, 97, 108, 85, 110, 108, 111, 99,
  107, 0, 0, 249, 1, 71, 108, 111, 98, 97, 108,
  76, 111, 99, 107, 0, 0, 238, 1, 71, 108, 111,
  98, 97, 108, 65, 108, 108, 111, 99, 0, 175, 0,
  69, 120, 105, 116, 80, 114, 111, 99, 101, 115, 115,
  0, 119, 1, 71, 101, 116, 77, 111, 100, 117, 108,
  101, 72, 97, 110, 100, 108, 101, 65, 0, 0, 75,
  69, 82, 78, 69, 76, 51, 50, 46, 100, 108, 108,
  0, 0, 58, 2, 83, 101, 110, 100, 77, 101, 115,
  115, 97, 103, 101, 65, 0, 0, 66, 0, 67, 108,
  111, 115, 101, 67, 108, 105, 112, 98, 111, 97, 114,
  100, 0, 0, 73, 2, 83, 101, 116, 67, 108, 105,
  112, 98, 111, 97, 114, 100, 68, 97, 116, 97, 0,
  0, 193, 0, 69, 109, 112, 116, 121, 67, 108, 105,
  112, 98, 111, 97, 114, 100, 0, 0, 245, 1, 79,
  112, 101, 110, 67, 108, 105, 112, 98, 111, 97, 114,
  100, 0, 82, 2, 83, 101, 116, 68, 108, 103, 73,
  116, 101, 109, 84, 101, 120, 116, 65, 0, 213, 2,
  119, 115, 112, 114, 105, 110, 116, 102, 65, 0, 53,
  2, 83, 101, 110, 100, 68, 108, 103, 73, 116, 101,
  109, 77, 101, 115, 115, 97, 103, 101, 65, 0, 3,
  2, 80, 111, 115, 116, 81, 117, 105, 116, 77, 101,
  115, 115, 97, 103, 101, 0, 198, 0, 69, 110, 100,
  68, 105, 97, 108, 111, 103, 0, 158, 0, 68, 105,
  97, 108, 111, 103, 66, 111, 120, 80, 97, 114, 97,
  109, 65, 0, 85, 83, 69, 82, 51, 50, 46, 100,
  108, 108, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 64, 66, 64, 0, 127, 0, 0, 0, 48,
  66, 64, 0, 54, 0, 0, 0, 32, 66, 64, 0,
  28, 0, 0, 0, 16, 66, 64, 0, 1, 0, 0,
  0, 4, 66, 64, 0, 43, 0, 0, 0, 244, 65,
  64, 0, 77, 0, 0, 0, 228, 65, 64, 0, 44,
  0, 0, 0, 216, 65, 64, 0, 45, 0, 0, 0,
  204, 65, 64, 0, 35, 0, 0, 0, 188, 65, 64,
  0, 69, 0, 0, 0, 172, 65, 64, 0, 2, 0,
  0, 0, 156, 65, 64, 0, 3, 0, 0, 0, 140,
  65, 64, 0, 4, 0, 0, 0, 124, 65, 64, 0,
  26, 0, 0, 0, 108, 65, 64, 0, 5, 0, 0,
  0, 96, 65, 64, 0, 6, 0, 0, 0, 84, 65,
  64, 0, 101, 0, 0, 0, 72, 65, 64, 0, 19,
  0, 0, 0, 60, 65, 64, 0, 9, 0, 0, 0,
  44, 65, 64, 0, 37, 0, 0, 0, 28, 65, 64,
  0, 56, 0, 0, 0, 12, 65, 64, 0, 41, 0,
  0, 0, 0, 65, 64, 0, 11, 0, 0, 0, 240,
  64, 64, 0, 12, 0, 0, 0, 228, 64, 64, 0,
  86, 0, 0, 0, 212, 64, 64, 0, 55, 0, 0,
  0, 196, 64, 64, 0, 7, 0, 0, 0, 184, 64,
  64, 0, 8, 0, 0, 0, 172, 64, 64, 0, 71,
  0, 0, 0, 156, 64, 64, 0, 13, 0, 0, 0,
  144, 64, 64, 0, 57, 0, 0, 0, 132, 64, 64,
  0, 14, 0, 0, 0, 116, 64, 64, 0, 15, 0,
  0, 0, 100, 64, 64, 0, 33, 0, 0, 0, 84,
  64, 64, 0, 16, 0, 0, 0, 68, 64, 64, 0,
  17, 0, 0, 0, 52, 64, 64, 0, 75, 0, 0,
  0, 36, 64, 64, 0, 96, 0, 0, 0, 20, 64,
  64, 0, 63, 0, 0, 0, 8, 64, 64, 0, 87,
  0, 0, 0, 248, 63, 64, 0, 18, 0, 0, 0,
  236, 63, 64, 0, 64, 0, 0, 0, 224, 63, 64,
  0, 38, 0, 0, 0, 208, 63, 64, 0, 39, 0,
  0, 0, 192, 63, 64, 0, 47, 0, 0, 0, 176,
  63, 64, 0, 62, 0, 0, 0, 164, 63, 64, 0,
  76, 0, 0, 0, 148, 63, 64, 0, 88, 0, 0,
  0, 132, 63, 64, 0, 78, 0, 0, 0, 116, 63,
  64, 0, 80, 0, 0, 0, 100, 63, 64, 0, 97,
  0, 0, 0, 88, 63, 64, 0, 20, 0, 0, 0,
  72, 63, 64, 0, 72, 0, 0, 0, 60, 63, 64,
  0, 21, 0, 0, 0, 48, 63, 64, 0, 22, 0,
  0, 0, 32, 63, 64, 0, 70, 0, 0, 0, 16,
  63, 64, 0, 24, 0, 0, 0, 0, 63, 64, 0,
  25, 0, 0, 0, 240, 62, 64, 0, 79, 0, 0,
  0, 224, 62, 64, 0, 26, 0, 0, 0, 208, 62,
  64, 0, 89, 0, 0, 0, 196, 62, 64, 0, 27,
  0, 0, 0, 184, 62, 64, 0, 36, 0, 0, 0,
  168, 62, 64, 0, 10, 0, 0, 0, 152, 62, 64,
  0, 65, 0, 0, 0, 136, 62, 64, 0, 29, 0,
  0, 0, 120, 62, 64, 0, 90, 0, 0, 0, 108,
  62, 64, 0, 73, 0, 0, 0, 96, 62, 64, 0,
  68, 0, 0, 0, 84, 62, 64, 0, 74, 0, 0,
  0, 72, 62, 64, 0, 30, 0, 0, 0, 60, 62,
  64, 0, 31, 0, 0, 0, 44, 62, 64, 0, 34,
  0, 0, 0, 28, 62, 64, 0, 32, 0, 0, 0,
  16, 62, 64, 0, 67, 0, 0, 0, 4, 62, 64,
  0, 42, 0, 0, 0, 244, 61, 64, 0, 0, 0,
  0, 0, 228, 61, 64, 0, 1, 0, 0, 0, 212,
  61, 64, 0, 2, 0, 0, 0, 192, 61, 64, 0,
  1, 0, 0, 0, 164, 61, 64, 0, 2, 0, 0,
  0, 144, 61, 64, 0, 3, 0, 0, 0, 120, 61,
  64, 0, 4, 0, 0, 0, 96, 61, 64, 0, 5,
  0, 0, 0, 72, 61, 64, 0, 6, 0, 0, 0,
  48, 61, 64, 0, 7, 0, 0, 0, 24, 61, 64,
  0, 8, 0, 0, 0, 4, 61, 64, 0, 9, 0,
  0, 0, 236, 60, 64, 0, 10, 0, 0, 0, 212,
  60, 64, 0, 11, 0, 0, 0, 188, 60, 64, 0,
  12, 0, 0, 0, 164, 60, 64, 0, 13, 0, 0,
  0, 140, 60, 64, 0, 14, 0, 0, 0, 120, 60,
  64, 0, 15, 0, 0, 0, 96, 60, 64, 0, 16,
  0, 0, 0, 72, 60, 64, 0, 1, 0, 0, 0,
  52, 60, 64, 0, 2, 0, 0, 0, 28, 60, 64,
  0, 1, 0, 0, 0, 0, 60, 64, 0, 2, 0,
  0, 0, 228, 59, 64, 0, 3, 0, 0, 0, 200,
  59, 64, 0, 4, 0, 0, 0, 172, 59, 64, 0,
  5, 0, 0, 0, 148, 59, 64, 0, 1, 0, 0,
  0, 132, 59, 64, 0, 2, 0, 0, 0, 108, 59,
  64, 0, 1, 0, 0, 0, 88, 59, 64, 0, 2,
  0, 0, 0, 68, 59, 64, 0, 3, 0, 0, 0,
  48, 59, 64, 0, 4, 0, 0, 0, 28, 59, 64,
  0, 5, 0, 0, 0, 8, 59, 64, 0, 6, 0,
  0, 0, 240, 58, 64, 0, 7, 0, 0, 0, 208,
  58, 64, 0, 8, 0, 0, 0, 184, 58, 64, 0,
  9, 0, 0, 0, 156, 58, 64, 0, 10, 0, 0,
  0, 132, 58, 64, 0, 11, 0, 0, 0, 104, 58,
  64, 0, 12, 0, 0, 0, 76, 58, 64, 0, 13,
  0, 0, 0, 48, 58, 64, 0, 1, 0, 0, 0,
  32, 58, 64, 0, 2, 0, 0, 0, 8, 58, 64,
  0, 3, 0, 0, 0, 240, 57, 64, 0, 4, 0,
  0, 0, 216, 57, 64, 0, 5, 0, 0, 0, 188,
  57, 64, 0, 6, 0, 0, 0, 164, 57, 64, 0,
  1, 0, 0, 0, 148, 57, 64, 0, 2, 0, 0,
  0, 124, 57, 64, 0, 3, 0, 0, 0, 100, 57,
  64, 0, 4, 0, 0, 0, 72, 57, 64, 0, 5,
  0, 0, 0, 40, 57, 64, 0, 1, 0, 0, 0,
  24, 57, 64, 0, 2, 0, 0, 0, 0, 57, 64,
  0, 2, 0, 0, 0, 232, 56, 64, 0, 1, 0,
  0, 0, 216, 56, 64, 0, 1, 0, 0, 0, 196,
  56, 64, 0, 1, 0, 0, 0, 172, 56, 64, 0,
  2, 0, 0, 0, 140, 56, 64, 0, 2, 0, 0,
  0, 116, 56, 64, 0, 1, 0, 0, 0, 88, 56,
  64, 0, 2, 0, 0, 0, 60, 56, 64, 0, 2,
  0, 0, 0, 40, 56, 64, 0, 1, 0, 0, 0,
  8, 56, 64, 0, 2, 0, 0, 0, 240, 55, 64,
  0, 3, 0, 0, 0, 212, 55, 64, 0, 1, 0,
  0, 0, 196, 55, 64, 0, 2, 0, 0, 0, 172,
  55, 64, 0, 3, 0, 0, 0, 148, 55, 64, 0,
  4, 0, 0, 0, 120, 55, 64, 0, 5, 0, 0,
  0, 92, 55, 64, 0, 6, 0, 0, 0, 68, 55,
  64, 0, 7, 0, 0, 0, 32, 55, 64, 0, 8,
  0, 0, 0, 4, 55, 64, 0, 9, 0, 0, 0,
  232, 54, 64, 0, 10, 0, 0, 0, 208, 54, 64,
  0, 11, 0, 0, 0, 180, 54, 64, 0, 12, 0,
  0, 0, 156, 54, 64, 0, 13, 0, 0, 0, 132,
  54, 64, 0, 14, 0, 0, 0, 108, 54, 64, 0,
  15, 0, 0, 0, 80, 54, 64, 0, 16, 0, 0,
  0, 56, 54, 64, 0, 17, 0, 0, 0, 28, 54,
  64, 0, 18, 0, 0, 0, 0, 54, 64, 0, 19,
  0, 0, 0, 228, 53, 64, 0, 20, 0, 0, 0,
  200, 53, 64, 0, 1, 0, 0, 0, 184, 53, 64,
  0, 2, 0, 0, 0, 160, 53, 64, 0, 1, 0,
  0, 0, 136, 53, 64, 0, 2, 0, 0, 0, 116,
  53, 64, 0, 1, 0, 0, 0, 96, 53, 64, 0,
  2, 0, 0, 0, 72, 53, 64, 0, 83, 85, 66,
  76, 65, 78, 71, 95, 85, 90, 66, 69, 75, 95,
  67, 89, 82, 73, 76, 76, 73, 67, 0, 0, 83,
  85, 66, 76, 65, 78, 71, 95, 85, 90, 66, 69,
  75, 95, 76, 65, 84, 73, 78, 0, 83, 85, 66,
  76, 65, 78, 71, 95, 85, 82, 68, 85, 95, 73,
  78, 68, 73, 65, 0, 0, 83, 85, 66, 76, 65,
  78, 71, 95, 85, 82, 68, 85, 95, 80, 65, 75,
  73, 83, 84, 65, 78, 0, 0, 0, 83, 85, 66,
  76, 65, 78, 71, 95, 83, 87, 69, 68, 73, 83,
  72, 95, 70, 73, 78, 76, 65, 78, 68, 0, 83,
  85, 66, 76, 65, 78, 71, 95, 83, 87, 69, 68,
  73, 83, 72, 0, 83, 85, 66, 76, 65, 78, 71,
  95, 83, 80, 65, 78, 73, 83, 72, 95, 80, 85,
  69, 82, 84, 79, 95, 82, 73, 67, 79, 0, 83,
  85, 66, 76, 65, 78, 71, 95, 83, 80, 65, 78,
  73, 83, 72, 95, 78, 73, 67, 65, 82, 65, 71,
  85, 65, 0, 0, 0, 83, 85, 66, 76, 65, 78,
  71, 95, 83, 80, 65, 78, 73, 83, 72, 95, 72,
  79, 78, 68, 85, 82, 65, 83, 0, 0, 0, 0,
  83, 85, 66, 76, 65, 78, 71, 95, 83, 80, 65,
  78, 73, 83, 72, 95, 69, 76, 95, 83, 65, 76,
  86, 65, 68, 79, 82, 0, 83, 85, 66, 76, 65,
  78, 71, 95, 83, 80, 65, 78, 73, 83, 72, 95,
  66, 79, 76, 73, 86, 73, 65, 0, 83, 85, 66,
  76, 65, 78, 71, 95, 83, 80, 65, 78, 73, 83,
  72, 95, 80, 65, 82, 65, 71, 85, 65, 89, 0,
  0, 0, 0, 83, 85, 66, 76, 65, 78, 71, 95,
  83, 80, 65, 78, 73, 83, 72, 95, 85, 82, 85,
  71, 85, 65, 89, 0, 83, 85, 66, 76, 65, 78,
  71, 95, 83, 80, 65, 78, 73, 83, 72, 95, 67,
  72, 73, 76, 69, 0, 0, 0, 83, 85, 66, 76,
  65, 78, 71, 95, 83, 80, 65, 78, 73, 83, 72,
  95, 69, 67, 85, 65, 68, 79, 82, 0, 83, 85,
  66, 76, 65, 78, 71, 95, 83, 80, 65, 78, 73,
  83, 72, 95, 65, 82, 71, 69, 78, 84, 73, 78,
  65, 0, 0, 0, 83, 85, 66, 76, 65, 78, 71,
  95, 83, 80, 65, 78, 73, 83, 72, 95, 80, 69,
  82, 85, 0, 0, 0, 0, 83, 85, 66, 76, 65,
  78, 71, 95, 83, 80, 65, 78, 73, 83, 72, 95,
  67, 79, 76, 79, 77, 66, 73, 65, 0, 0, 0,
  0, 83, 85, 66, 76, 65, 78, 71, 95, 83, 80,
  65, 78, 73, 83, 72, 95, 86, 69, 78, 69, 90,
  85, 69, 76, 65, 0, 0, 0, 83, 85, 66, 76,
  65, 78, 71, 95, 83, 80, 65, 78, 73, 83, 72,
  95, 68, 79, 77, 73, 78, 73, 67, 65, 78, 95,
  82, 69, 80, 85, 66, 76, 73, 67, 0, 0, 83,
  85, 66, 76, 65, 78, 71, 95, 83, 80, 65, 78,
  73, 83, 72, 95, 80, 65, 78, 65, 77, 65, 0,
  0, 83, 85, 66, 76, 65, 78, 71, 95, 83, 80,
  65, 78, 73, 83, 72, 95, 67, 79, 83, 84, 65,
  95, 82, 73, 67, 65, 0, 0, 83, 85, 66, 76,
  65, 78, 71, 95, 83, 80, 65, 78, 73, 83, 72,
  95, 71, 85, 65, 84, 69, 77, 65, 76, 65, 0,
  0, 0, 83, 85, 66, 76, 65, 78, 71, 95, 83,
  80, 65, 78, 73, 83, 72, 95, 77, 79, 68, 69,
  82, 78, 0, 0, 83, 85, 66, 76, 65, 78, 71,
  95, 83, 80, 65, 78, 73, 83, 72, 95, 77, 69,
  88, 73, 67, 65, 78, 0, 83, 85, 66, 76, 65,
  78, 71, 95, 83, 80, 65, 78, 73, 83, 72, 0,
  83, 85, 66, 76, 65, 78, 71, 95, 83, 69, 82,
  66, 73, 65, 78, 95, 67, 89, 82, 73, 76, 76,
  73, 67, 0, 0, 0, 0, 83, 85, 66, 76, 65,
  78, 71, 95, 83, 69, 82, 66, 73, 65, 78, 95,
  76, 65, 84, 73, 78, 0, 0, 0, 83, 85, 66,
  76, 65, 78, 71, 95, 80, 79, 82, 84, 85, 71,
  85, 69, 83, 69, 95, 66, 82, 65, 90, 73, 76,
  73, 65, 78, 0, 0, 0, 0, 83, 85, 66, 76,
  65, 78, 71, 95, 80, 79, 82, 84, 85, 71, 85,
  69, 83, 69, 0, 0, 83, 85, 66, 76, 65, 78,
  71, 95, 78, 79, 82, 87, 69, 71, 73, 65, 78,
  95, 78, 89, 78, 79, 82, 83, 75, 0, 0, 0,
  83, 85, 66, 76, 65, 78, 71, 95, 78, 79, 82,
  87, 69, 71, 73, 65, 78, 95, 66, 79, 75, 77,
  65, 76, 0, 0, 0, 0, 83, 85, 66, 76, 65,
  78, 71, 95, 78, 69, 80, 65, 76, 73, 95, 73,
  78, 68, 73, 65, 0, 0, 0, 0, 83, 85, 66,
  76, 65, 78, 71, 95, 77, 65, 76, 65, 89, 95,
  66, 82, 85, 78, 69, 73, 95, 68, 65, 82, 85,
  83, 83, 65, 76, 65, 77, 0, 83, 85, 66, 76,
  65, 78, 71, 95, 77, 65, 76, 65, 89, 95, 77,
  65, 76, 65, 89, 83, 73, 65, 0, 0, 83, 85,
  66, 76, 65, 78, 71, 95, 76, 73, 84, 72, 85,
  65, 78, 73, 65, 78, 0, 0, 83, 85, 66, 76,
  65, 78, 71, 95, 75, 79, 82, 69, 65, 78, 0,
  0, 83, 85, 66, 76, 65, 78, 71, 95, 75, 65,
  83, 72, 77, 73, 82, 73, 95, 73, 78, 68, 73,
  65, 0, 0, 83, 85, 66, 76, 65, 78, 71, 95,
  73, 84, 65, 76, 73, 65, 78, 95, 83, 87, 73,
  83, 83, 0, 0, 0, 83, 85, 66, 76, 65, 78,
  71, 95, 73, 84, 65, 76, 73, 65, 78, 0, 83,
  85, 66, 76, 65, 78, 71, 95, 71, 69, 82, 77,
  65, 78, 95, 76, 73, 69, 67, 72, 84, 69, 78,
  83, 84, 69, 73, 78, 0, 0, 0, 0, 83, 85,
  66, 76, 65, 78, 71, 95, 71, 69, 82, 77, 65,
  78, 95, 76, 85, 88, 69, 77, 66, 79, 85, 82,
  71, 0, 0, 0, 83, 85, 66, 76, 65, 78, 71,
  95, 71, 69, 82, 77, 65, 78, 95, 65, 85, 83,
  84, 82, 73, 65, 78, 0, 83, 85, 66, 76, 65,
  78, 71, 95, 71, 69, 82, 77, 65, 78, 95, 83,
  87, 73, 83, 83, 0, 0, 0, 0, 83, 85, 66,
  76, 65, 78, 71, 95, 71, 69, 82, 77, 65, 78,
  0, 0, 83, 85, 66, 76, 65, 78, 71, 95, 70,
  82, 69, 78, 67, 72, 95, 77, 79, 78, 65, 67,
  79, 0, 0, 0, 83, 85, 66, 76, 65, 78, 71,
  95, 70, 82, 69, 78, 67, 72, 95, 76, 85, 88,
  69, 77, 66, 79, 85, 82, 71, 0, 0, 0, 83,
  85, 66, 76, 65, 78, 71, 95, 70, 82, 69, 78,
  67, 72, 95, 83, 87, 73, 83, 83, 0, 0, 0,
  0, 83, 85, 66, 76, 65, 78, 71, 95, 70, 82,
  69, 78, 67, 72, 95, 67, 65, 78, 65, 68, 73,
  65, 78, 0, 83, 85, 66, 76, 65, 78, 71, 95,
  70, 82, 69, 78, 67, 72, 95, 66, 69, 76, 71,
  73, 65, 78, 0, 0, 83, 85, 66, 76, 65, 78,
  71, 95, 70, 82, 69, 78, 67, 72, 0, 0, 83,
  85, 66, 76, 65, 78, 71, 95, 69, 78, 71, 76,
  73, 83, 72, 95, 80, 72, 73, 76, 73, 80, 80,
  73, 78, 69, 83, 0, 83, 85, 66, 76, 65, 78,
  71, 95, 69, 78, 71, 76, 73, 83, 72, 95, 90,
  73, 77, 66, 65, 66, 87, 69, 0, 0, 0, 0,
  83, 85, 66, 76, 65, 78, 71, 95, 69, 78, 71,
  76, 73, 83, 72, 95, 84, 82, 73, 78, 73, 68,
  65, 68, 0, 0, 0, 0, 83, 85, 66, 76, 65,
  78, 71, 95, 69, 78, 71, 76, 73, 83, 72, 95,
  66, 69, 76, 73, 90, 69, 0, 0, 83, 85, 66,
  76, 65, 78, 71, 95, 69, 78, 71, 76, 73, 83,
  72, 95, 67, 65, 82, 73, 66, 66, 69, 65, 78,
  0, 0, 0, 83, 85, 66, 76, 65, 78, 71, 95,
  69, 78, 71, 76, 73, 83, 72, 95, 74, 65, 77,
  65, 73, 67, 65, 0, 83, 85, 66, 76, 65, 78,
  71, 95, 69, 78, 71, 76, 73, 83, 72, 95, 83,
  79, 85, 84, 72, 95, 65, 70, 82, 73, 67, 65,
  0, 0, 0, 0, 83, 85, 66, 76, 65, 78, 71,
  95, 69, 78, 71, 76, 73, 83, 72, 95, 69, 73,
  82, 69, 0, 0, 0, 0, 83, 85, 66, 76, 65,
  78, 71, 95, 69, 78, 71, 76, 73, 83, 72, 95,
  78, 90, 0, 0, 83, 85, 66, 76, 65, 78, 71,
  95, 69, 78, 71, 76, 73, 83, 72, 95, 67, 65,
  78, 0, 83, 85, 66, 76, 65, 78, 71, 95, 69,
  78, 71, 76, 73, 83, 72, 95, 65, 85, 83, 0,
  83, 85, 66, 76, 65, 78, 71, 95, 69, 78, 71,
  76, 73, 83, 72, 95, 85, 75, 0, 0, 83, 85,
  66, 76, 65, 78, 71, 95, 69, 78, 71, 76, 73,
  83, 72, 95, 85, 83, 0, 0, 83, 85, 66, 76,
  65, 78, 71, 95, 68, 85, 84, 67, 72, 95, 66,
  69, 76, 71, 73, 65, 78, 0, 0, 0, 83, 85,
  66, 76, 65, 78, 71, 95, 68, 85, 84, 67, 72,
  0, 0, 0, 83, 85, 66, 76, 65, 78, 71, 95,
  67, 72, 73, 78, 69, 83, 69, 95, 77, 65, 67,
  65, 85, 0, 0, 0, 83, 85, 66, 76, 65, 78,
  71, 95, 67, 72, 73, 78, 69, 83, 69, 95, 83,
  73, 78, 71, 65, 80, 79, 82, 69, 0, 0, 0,
  83, 85, 66, 76, 65, 78, 71, 95, 67, 72, 73,
  78, 69, 83, 69, 95, 72, 79, 78, 71, 75, 79,
  78, 71, 0, 0, 0, 0, 83, 85, 66, 76, 65,
  78, 71, 95, 67, 72, 73, 78, 69, 83, 69, 95,
  83, 73, 77, 80, 76, 73, 70, 73, 69, 68, 0,
  0, 83, 85, 66, 76, 65, 78, 71, 95, 67, 72,
  73, 78, 69, 83, 69, 95, 84, 82, 65, 68, 73,
  84, 73, 79, 78, 65, 76, 0, 83, 85, 66, 76,
  65, 78, 71, 95, 65, 90, 69, 82, 73, 95, 67,
  89, 82, 73, 76, 76, 73, 67, 0, 0, 83, 85,
  66, 76, 65, 78, 71, 95, 65, 90, 69, 82, 73,
  95, 76, 65, 84, 73, 78, 0, 83, 85, 66, 76,
  65, 78, 71, 95, 65, 82, 65, 66, 73, 67, 95,
  81, 65, 84, 65, 82, 0, 0, 0, 0, 83, 85,
  66, 76, 65, 78, 71, 95, 65, 82, 65, 66, 73,
  67, 95, 66, 65, 72, 82, 65, 73, 78, 0, 0,
  83, 85, 66, 76, 65, 78, 71, 95, 65, 82, 65,
  66, 73, 67, 95, 85, 65, 69, 0, 0, 83, 85,
  66, 76, 65, 78, 71, 95, 65, 82, 65, 66, 73,
  67, 95, 75, 85, 87, 65, 73, 84, 0, 0, 0,
  83, 85, 66, 76, 65, 78, 71, 95, 65, 82, 65,
  66, 73, 67, 95, 76, 69, 66, 65, 78, 79, 78,
  0, 0, 83, 85, 66, 76, 65, 78, 71, 95, 65,
  82, 65, 66, 73, 67, 95, 74, 79, 82, 68, 65,
  78, 0, 0, 0, 83, 85, 66, 76, 65, 78, 71,
  95, 65, 82, 65, 66, 73, 67, 95, 83, 89, 82,
  73, 65, 0, 0, 0, 0, 83, 85, 66, 76, 65,
  78, 71, 95, 65, 82, 65, 66, 73, 67, 95, 89,
  69, 77, 69, 78, 0, 0, 0, 0, 83, 85, 66,
  76, 65, 78, 71, 95, 65, 82, 65, 66, 73, 67,
  95, 79, 77, 65, 78, 0, 83, 85, 66, 76, 65,
  78, 71, 95, 65, 82, 65, 66, 73, 67, 95, 84,
  85, 78, 73, 83, 73, 65, 0, 0, 83, 85, 66,
  76, 65, 78, 71, 95, 65, 82, 65, 66, 73, 67,
  95, 77, 79, 82, 79, 67, 67, 79, 0, 0, 83,
  85, 66, 76, 65, 78, 71, 95, 65, 82, 65, 66,
  73, 67, 95, 65, 76, 71, 69, 82, 73, 65, 0,
  0, 83, 85, 66, 76, 65, 78, 71, 95, 65, 82,
  65, 66, 73, 67, 95, 76, 73, 66, 89, 65, 0,
  0, 0, 0, 83, 85, 66, 76, 65, 78, 71, 95,
  65, 82, 65, 66, 73, 67, 95, 69, 71, 89, 80,
  84, 0, 0, 0, 0, 83, 85, 66, 76, 65, 78,
  71, 95, 65, 82, 65, 66, 73, 67, 95, 73, 82,
  65, 81, 0, 83, 85, 66, 76, 65, 78, 71, 95,
  65, 82, 65, 66, 73, 67, 95, 83, 65, 85, 68,
  73, 95, 65, 82, 65, 66, 73, 65, 0, 83, 85,
  66, 76, 65, 78, 71, 95, 83, 89, 83, 95, 68,
  69, 70, 65, 85, 76, 84, 0, 83, 85, 66, 76,
  65, 78, 71, 95, 68, 69, 70, 65, 85, 76, 84,
  0, 83, 85, 66, 76, 65, 78, 71, 95, 78, 69,
  85, 84, 82, 65, 76, 0, 76, 65, 78, 71, 95,
  86, 73, 69, 84, 78, 65, 77, 69, 83, 69, 0,
  76, 65, 78, 71, 95, 85, 90, 66, 69, 75, 0,
  0, 76, 65, 78, 71, 95, 85, 82, 68, 85, 0,
  0, 0, 76, 65, 78, 71, 95, 85, 75, 82, 65,
  73, 78, 73, 65, 78, 0, 0, 76, 65, 78, 71,
  95, 84, 85, 82, 75, 73, 83, 72, 0, 0, 0,
  0, 76, 65, 78, 71, 95, 84, 72, 65, 73, 0,
  0, 0, 76, 65, 78, 71, 95, 84, 69, 76, 85,
  71, 85, 0, 76, 65, 78, 71, 95, 84, 65, 84,
  65, 82, 0, 0, 76, 65, 78, 71, 95, 84, 65,
  77, 73, 76, 0, 0, 76, 65, 78, 71, 95, 83,
  89, 82, 73, 65, 67, 0, 76, 65, 78, 71, 95,
  83, 87, 69, 68, 73, 83, 72, 0, 0, 0, 0,
  76, 65, 78, 71, 95, 83, 87, 65, 72, 73, 76,
  73, 0, 0, 0, 0, 76, 65, 78, 71, 95, 83,
  80, 65, 78, 73, 83, 72, 0, 0, 0, 0, 76,
  65, 78, 71, 95, 83, 76, 79, 86, 69, 78, 73,
  65, 78, 0, 0, 76, 65, 78, 71, 95, 83, 76,
  79, 86, 65, 75, 0, 76, 65, 78, 71, 95, 83,
  73, 78, 68, 72, 73, 0, 76, 65, 78, 71, 95,
  83, 69, 82, 66, 73, 65, 78, 0, 0, 0, 0,
  76, 65, 78, 71, 95, 83, 65, 78, 83, 75, 82,
  73, 84, 0, 0, 0, 76, 65, 78, 71, 95, 82,
  85, 83, 83, 73, 65, 78, 0, 0, 0, 0, 76,
  65, 78, 71, 95, 82, 79, 77, 65, 78, 73, 65,
  78, 0, 0, 0, 76, 65, 78, 71, 95, 80, 85,
  78, 74, 65, 66, 73, 0, 0, 0, 0, 76, 65,
  78, 71, 95, 80, 79, 82, 84, 85, 71, 85, 69,
  83, 69, 0, 76, 65, 78, 71, 95, 80, 79, 76,
  73, 83, 72, 0, 76, 65, 78, 71, 95, 79, 82,
  73, 89, 65, 0, 0, 76, 65, 78, 71, 95, 78,
  79, 82, 87, 69, 71, 73, 65, 78, 0, 0, 76,
  65, 78, 71, 95, 78, 69, 80, 65, 76, 73, 0,
  76, 65, 78, 71, 95, 77, 79, 78, 71, 79, 76,
  73, 65, 78, 0, 0, 76, 65, 78, 71, 95, 77,
  65, 82, 65, 84, 72, 73, 0, 0, 0, 0, 76,
  65, 78, 71, 95, 77, 65, 78, 73, 80, 85, 82,
  73, 0, 0, 0, 76, 65, 78, 71, 95, 77, 65,
  76, 65, 89, 65, 76, 65, 77, 0, 0, 76, 65,
  78, 71, 95, 77, 65, 76, 65, 89, 0, 0, 76,
  65, 78, 71, 95, 77, 65, 67, 69, 68, 79, 78,
  73, 65, 78, 0, 76, 65, 78, 71, 95, 76, 73,
  84, 72, 85, 65, 78, 73, 65, 78, 0, 76, 65,
  78, 71, 95, 76, 65, 84, 86, 73, 65, 78, 0,
  0, 0, 0, 76, 65, 78, 71, 95, 75, 89, 82,
  71, 89, 90, 0, 76, 65, 78, 71, 95, 75, 79,
  82, 69, 65, 78, 0, 76, 65, 78, 71, 95, 75,
  79, 78, 75, 65, 78, 73, 0, 0, 0, 0, 76,
  65, 78, 71, 95, 75, 65, 90, 65, 75, 0, 0,
  76, 65, 78, 71, 95, 75, 65, 83, 72, 77, 73,
  82, 73, 0, 0, 0, 76, 65, 78, 71, 95, 75,
  65, 78, 78, 65, 68, 65, 0, 0, 0, 0, 76,
  65, 78, 71, 95, 74, 65, 80, 65, 78, 69, 83,
  69, 0, 0, 0, 76, 65, 78, 71, 95, 73, 84,
  65, 76, 73, 65, 78, 0, 0, 0, 0, 76, 65,
  78, 71, 95, 73, 78, 68, 79, 78, 69, 83, 73,
  65, 78, 0, 76, 65, 78, 71, 95, 73, 67, 69,
  76, 65, 78, 68, 73, 67, 0, 0, 76, 65, 78,
  71, 95, 72, 85, 78, 71, 65, 82, 73, 65, 78,
  0, 0, 76, 65, 78, 71, 95, 72, 73, 78, 68,
  73, 0, 0, 76, 65, 78, 71, 95, 72, 69, 66,
  82, 69, 87, 0, 76, 65, 78, 71, 95, 71, 85,
  74, 65, 82, 65, 84, 73, 0, 0, 0, 76, 65,
  78, 71, 95, 71, 82, 69, 69, 75, 0, 0, 76,
  65, 78, 71, 95, 71, 69, 82, 77, 65, 78, 0,
  76, 65, 78, 71, 95, 71, 69, 79, 82, 71, 73,
  65, 78, 0, 0, 0, 76, 65, 78, 71, 95, 71,
  65, 76, 73, 67, 73, 65, 78, 0, 0, 0, 76,
  65, 78, 71, 95, 70, 82, 69, 78, 67, 72, 0,
  76, 65, 78, 71, 95, 70, 73, 78, 78, 73, 83,
  72, 0, 0, 0, 0, 76, 65, 78, 71, 95, 70,
  65, 82, 83, 73, 0, 0, 76, 65, 78, 71, 95,
  70, 65, 69, 82, 79, 69, 83, 69, 0, 0, 0,
  76, 65, 78, 71, 95, 69, 83, 84, 79, 78, 73,
  65, 78, 0, 0, 0, 76, 65, 78, 71, 95, 69,
  78, 71, 76, 73, 83, 72, 0, 0, 0, 0, 76,
  65, 78, 71, 95, 68, 85, 84, 67, 72, 0, 0,
  76, 65, 78, 71, 95, 68, 73, 86, 69, 72, 73,
  0, 76, 65, 78, 71, 95, 68, 65, 78, 73, 83,
  72, 0, 76, 65, 78, 71, 95, 67, 90, 69, 67,
  72, 0, 0, 76, 65, 78, 71, 95, 67, 82, 79,
  65, 84, 73, 65, 78, 0, 0, 0, 76, 65, 78,
  71, 95, 67, 72, 73, 78, 69, 83, 69, 0, 0,
  0, 0, 76, 65, 78, 71, 95, 67, 65, 84, 65,
  76, 65, 78, 0, 0, 0, 0, 76, 65, 78, 71,
  95, 66, 85, 76, 71, 65, 82, 73, 65, 78, 0,
  0, 76, 65, 78, 71, 95, 66, 69, 78, 71, 65,
  76, 73, 0, 0, 0, 0, 76, 65, 78, 71, 95,
  66, 69, 76, 65, 82, 85, 83, 73, 65, 78, 0,
  76, 65, 78, 71, 95, 66, 65, 83, 81, 85, 69,
  0, 76, 65, 78, 71, 95, 65, 90, 69, 82, 73,
  0, 0, 76, 65, 78, 71, 95, 65, 83, 83, 65,
  77, 69, 83, 69, 0, 0, 0, 76, 65, 78, 71,
  95, 65, 82, 77, 69, 78, 73, 65, 78, 0, 0,
  0, 76, 65, 78, 71, 95, 65, 82, 65, 66, 73,
  67, 0, 76, 65, 78, 71, 95, 65, 76, 66, 65,
  78, 73, 65, 78, 0, 0, 0, 76, 65, 78, 71,
  95, 65, 70, 82, 73, 75, 65, 65, 78, 83, 0,
  0, 76, 65, 78, 71, 95, 73, 78, 86, 65, 82,
  73, 65, 78, 84, 0, 0, 76, 65, 78, 71, 95,
  78, 69, 85, 84, 82, 65, 76, 0, 0, 0, 0,
  37, 117, 0, 0, 76, 97, 110, 103, 117, 97, 103,
  101, 32, 73, 68, 58, 32, 37, 100, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 2, 0, 5, 0, 0,
  0, 32, 0, 0, 128, 24, 0, 0, 0, 56, 0,
  0, 128, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 1, 0, 101, 0, 0, 0,
  80, 0, 0, 128, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0,
  0, 0, 104, 0, 0, 128, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0,
  9, 4, 0, 0, 128, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  1, 0, 9, 4, 0, 0, 144, 0, 0, 0, 160,
  80, 0, 0, 230, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 136, 81, 0, 0, 25, 2, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 192, 8,
  200, 128, 0, 0, 0, 0, 5, 0, 0, 0, 0,
  0, 135, 0, 75, 0, 0, 0, 0, 0, 77, 0,
  97, 0, 107, 0, 101, 0, 76, 0, 97, 0, 110,
  0, 103, 0, 73, 0, 68, 0, 0, 0, 8, 0,
  77, 0, 83, 0, 32, 0, 83, 0, 97, 0, 110,
  0, 115, 0, 32, 0, 83, 0, 101, 0, 114, 0,
  105, 0, 102, 0, 0, 0, 0, 0, 3, 0, 33,
  80, 0, 0, 0, 0, 7, 0, 7, 0, 121, 0,
  100, 0, 234, 3, 255, 255, 133, 0, 0, 0, 0,
  0, 0, 0, 3, 0, 33, 80, 0, 0, 0, 0,
  7, 0, 24, 0, 121, 0, 100, 0, 233, 3, 255,
  255, 133, 0, 0, 0, 0, 0, 0, 0, 1, 0,
  1, 80, 0, 0, 0, 0, 78, 0, 54, 0, 50,
  0, 14, 0, 1, 0, 255, 255, 128, 0, 67, 0,
  111, 0, 112, 0, 121, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 1, 80, 0, 0, 0, 0, 7, 0,
  54, 0, 50, 0, 14, 0, 2, 0, 255, 255, 128,
  0, 69, 0, 120, 0, 105, 0, 116, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 2, 80, 0, 0, 0,
  0, 7, 0, 41, 0, 121, 0, 8, 0, 235, 3,
  255, 255, 130, 0, 0, 0, 0, 0, 0, 0, 60,
  63, 120, 109, 108, 32, 118, 101, 114, 115, 105, 111,
  110, 61, 34, 49, 46, 48, 34, 32, 101, 110, 99,
  111, 100, 105, 110, 103, 61, 34, 85, 84, 70, 45,
  56, 34, 32, 115, 116, 97, 110, 100, 97, 108, 111,
  110, 101, 61, 34, 121, 101, 115, 34, 63, 62, 13,
  10, 60, 97, 115, 115, 101, 109, 98, 108, 121, 32,
  120, 109, 108, 110, 115, 61, 34, 117, 114, 110, 58,
  115, 99, 104, 101, 109, 97, 115, 45, 109, 105, 99,
  114, 111, 115, 111, 102, 116, 45, 99, 111, 109, 58,
  97, 115, 109, 46, 118, 49, 34, 32, 109, 97, 110,
  105, 102, 101, 115, 116, 86, 101, 114, 115, 105, 111,
  110, 61, 34, 49, 46, 48, 34, 62, 13, 10, 60,
  97, 115, 115, 101, 109, 98, 108, 121, 73, 100, 101,
  110, 116, 105, 116, 121, 32, 118, 101, 114, 115, 105,
  111, 110, 61, 34, 49, 46, 48, 46, 48, 46, 48,
  34, 32, 112, 114, 111, 99, 101, 115, 115, 111, 114,
  65, 114, 99, 104, 105, 116, 101, 99, 116, 117, 114,
  101, 61, 34, 88, 56, 54, 34, 32, 110, 97, 109,
  101, 61, 34, 78, 117, 108, 108, 115, 111, 102, 116,
  46, 78, 83, 73, 83, 46, 77, 97, 107, 101, 76,
  97, 110, 103, 73, 100, 34, 32, 116, 121, 112, 101,
  61, 34, 119, 105, 110, 51, 50, 34, 47, 62, 13,
  10, 60, 100, 101, 115, 99, 114, 105, 112, 116, 105,
  111, 110, 62, 77, 97, 107, 101, 76, 97, 110, 103,
  73, 100, 60, 47, 100, 101, 115, 99, 114, 105, 112,
  116, 105, 111, 110, 62, 13, 10, 60, 100, 101, 112,
  101, 110, 100, 101, 110, 99, 121, 62, 13, 10, 60,
  100, 101, 112, 101, 110, 100, 101, 110, 116, 65, 115,
  115, 101, 109, 98, 108, 121, 62, 13, 10, 60, 97,
  115, 115, 101, 109, 98, 108, 121, 73, 100, 101, 110,
  116, 105, 116, 121, 32, 116, 121, 112, 101, 61, 34,
  119, 105, 110, 51, 50, 34, 32, 110, 97, 109, 101,
  61, 34, 77, 105, 99, 114, 111, 115, 111, 102, 116,
  46, 87, 105, 110, 100, 111, 119, 115, 46, 67, 111,
  109, 109, 111, 110, 45, 67, 111, 110, 116, 114, 111,
  108, 115, 34, 32, 118, 101, 114, 115, 105, 111, 110,
  61, 34, 54, 46, 48, 46, 48, 46, 48, 34, 32,
  112, 114, 111, 99, 101, 115, 115, 111, 114, 65, 114,
  99, 104, 105, 116, 101, 99, 116, 117, 114, 101, 61,
  34, 88, 56, 54, 34, 32, 112, 117, 98, 108, 105,
  99, 75, 101, 121, 84, 111, 107, 101, 110, 61, 34,
  54, 53, 57, 53, 98, 54, 52, 49, 52, 52, 99,
  99, 102, 49, 100, 102, 34, 32, 108, 97, 110, 103,
  117, 97, 103, 101, 61, 34, 42, 34, 32, 47, 62,
  13, 10, 60, 47, 100, 101, 112, 101, 110, 100, 101,
  110, 116, 65, 115, 115, 101, 109, 98, 108, 121, 62,
  13, 10, 60, 47, 100, 101, 112, 101, 110, 100, 101,
  110, 99, 121, 62, 13, 10, 60, 47, 97, 115, 115,
  101, 109, 98, 108, 121, 62, 13, 10, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0
};
